/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;

import java.io.File;
import java.io.IOException;
import java.util.List;

import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Story(MAVEN_CLIENT)
public class TestJarDependencyTestCase extends AbstractMavenClientTestCase {

  @Rule
  public ExpectedException expectedException = none();

  @Override
  protected void beforeTest() {}

  private MavenConfiguration createMavenConfiguration(File localRepository) {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .ignoreArtifactDescriptorRepositories(false)
        .build();
  }

  @Test
  public void classifierDeclaredInDependencyShouldTakePrecedence() throws IOException {
    expectedException
        .expectCause(hasCause(hasMessage(containsString("Could not find artifact org.mule.runtime:mule-core:jar:mule-plugin:4.1.5"))));
    doTestWith("pom-with-test-jar-dependency-explicit-classifier/pom.xml", true);
  }

  @Test
  public void resolveDependencyToTestJarArtifactWithScopeTest() throws IOException {
    // Will check also exclusion is correctly set when adding explicit dependency for test dependencies (which are handled manually)
    doTestWith("pom-with-test-jar-dependency-scope-test/pom.xml", true);
  }

  @Test
  public void resolveDependencyToTestJarArtifactWithScopeCompile() throws IOException {
    doTestWith("pom-with-test-jar-dependency-scope-compile/pom.xml", false);
  }

  private void doTestWith(String pomResourceFile, boolean includeTestDependencies) throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(temporaryFolder.newFolder()));
    File artifactFile = toFile(getClass().getClassLoader().getResource(pomResourceFile));
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, includeTestDependencies, false, empty(), empty());

    assertThat(bundleDependencies, hasSize(1));
    BundleDescriptor bundleDescriptor = bundleDependencies.get(0).getDescriptor();
    assertThat(bundleDescriptor.getArtifactId(), equalTo("mule-core"));
    assertThat(bundleDescriptor.getClassifier(), equalTo(of("tests")));
    assertThat(bundleDescriptor.getType(), equalTo("jar"));
  }

}
