/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Collections.newSetFromMap;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.either;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.internal.AetherMavenClient;
import org.mule.maven.test.ArtifactCreator;
import org.mule.maven.test.MavenRepository;

import java.util.IdentityHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import io.qameta.allure.Feature;
import org.hamcrest.BaseMatcher;
import org.hamcrest.Description;
import org.junit.Before;
import org.junit.Rule;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@Feature(MAVEN_CLIENT)
public abstract class AbstractMavenClientTestCase {

  protected static final String VERSION = "1.0";
  protected static final String GROUP_ID = "org.mule";
  protected static final String MULE_PLUGIN = AetherMavenClient.MULE_PLUGIN_CLASSIFIER;
  protected static final String RAML = "raml";
  protected static final String RAML_FRAGMENT = "raml-fragment";
  protected static final String OAS = "oas";
  protected static final String WSDL = "wsdl";

  protected MavenClient mavenClient;
  protected MavenClientProvider mavenClientProvider;

  @Rule
  public MavenRepository repositoryFolder = new MavenRepository();
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void setUp() throws Exception {
    mavenClientProvider = discoverProvider(getClass().getClassLoader());
    beforeTest();
  }

  protected abstract void beforeTest() throws Exception;


  protected ArtifactCreator.Builder<ArtifactCreator.Builder> artifact(String artifactId) {
    return ArtifactCreator.artifact(GROUP_ID, artifactId, VERSION);
  }

  protected ArtifactCreator.DependencyBuilder dependency(String artifactId) {
    return ArtifactCreator.dependency(GROUP_ID, artifactId, VERSION);
  }

  protected ArtifactCreator.ExclusionBuilder exclusion(String artifactId) {
    return ArtifactCreator.exclusion(GROUP_ID, artifactId);
  }

  protected BundleDescriptor getDescriptor(String artifactId) {
    return getDescriptor(artifactId, VERSION);
  }

  protected BundleDescriptor getDescriptor(String artifactId, String version) {
    return getDescriptor(artifactId, version, "pom");
  }

  protected BundleDescriptor getDescriptor(String artifactId, String version, String type) {
    return getDescriptor(artifactId, version, type, null);
  }

  protected BundleDescriptor getDescriptor(String artifactId, String version, String type, String classifier) {
    BundleDescriptor.Builder builder = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(artifactId)
        .setVersion(version)
        .setType(type);
    if (classifier != null) {
      builder.setClassifier(classifier);
    }
    return builder.build();
  }

  private List<BundleDependency> collectAllContextDependencies(List<BundleDependency> bundleDependencies) {
    List<BundleDependency> allBundleDependencies = new LinkedList<>();
    bundleDependencies
        .stream()
        .filter(bd -> !isPlugin(bd))
        .peek(allBundleDependencies::add)
        .forEach(bd -> allBundleDependencies.addAll(collectAllContextDependencies(bd.getTransitiveDependencies())));
    return allBundleDependencies;
  }

  private boolean isPlugin(BundleDependency bundleDependency) {
    return bundleDependency.getDescriptor().getClassifier().map(MULE_PLUGIN::equals).orElse(false);
  }

  private Set<BundleDependency> getInstancesSet(List<BundleDependency> dependencies) {
    Set<BundleDependency> dependenciesSet = newSetFromMap(new IdentityHashMap<>());
    dependenciesSet.addAll(dependencies);
    assertThat(dependenciesSet.size(), equalTo(dependencies.size()));
    return dependenciesSet;
  }

  /**
   * There should be only one instance for every dependency within the nonPlugin dependencies tree.
   * There should be only one instance for every dependency within every plugin dependency tree.
   * No instance should be in more than one subtree.
   */
  protected void checkNoDuplicatedInstances(List<BundleDependency> bundleDependencies) {
    List<List<BundleDependency>> subTrees = new LinkedList<>();

    List<BundleDependency> nonPlugins = new LinkedList<>();

    bundleDependencies.forEach(
                               bd -> {
                                 if (isPlugin(bd)) {
                                   subTrees.add(collectAllContextDependencies(bd.getTransitiveDependencies()));
                                 } else {
                                   //Non plugins should not be collected because they are already flattened in the first level.
                                   nonPlugins.add(bd);
                                 }
                               });

    subTrees.add(nonPlugins);

    List<Set<BundleDependency>> instancesSubtrees = subTrees.stream().map(this::getInstancesSet).collect(toList());

    for (Set<BundleDependency> subtree1 : instancesSubtrees) {
      for (Set<BundleDependency> subtree2 : instancesSubtrees) {
        if (subtree1 == subtree2) {
          continue;
        }
        Set<BundleDependency> intersection = newSetFromMap(new IdentityHashMap<>());
        intersection.addAll(subtree1);
        intersection.retainAll(subtree2);
        assertThat(intersection, either(iterableWithSize(0)).or(contains(PluginMatcher.plugins())));
      }
    }

  }

  private static class PluginMatcher extends BaseMatcher {

    public static PluginMatcher plugins() {
      return new PluginMatcher();
    }

    private PluginMatcher() {}

    @Override
    public void describeTo(Description description) {
      description.appendText("Expected plugin");
    }

    @Override
    public boolean matches(Object item) {
      if (item instanceof BundleDependency) {
        return ((BundleDependency) item).getDescriptor().getClassifier().map(MULE_PLUGIN::equals).orElse(false);
      }
      return false;
    }
  }

}
