/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsNot.not;
import static org.mule.maven.client.api.model.BundleScope.COMPILE;
import static org.mule.maven.client.api.model.BundleScope.TEST;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.BundleScope;
import org.mule.maven.client.api.model.MavenConfiguration;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(MAVEN_CLIENT)
public class ManageDependencyDeriverTestCase extends AbstractMavenClientTestCase {

  @Override
  protected void beforeTest() {}

  private MavenConfiguration createMavenConfiguration(File localRepository) {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localRepository)
        .ignoreArtifactDescriptorRepositories(false)
        .build();
  }

  @Test
  public void versionManaged() throws IOException {
    doTestWith("pom-manage-dependency-version-managed/pom.xml", bundleDependencies -> {
      assertThat(bundleDependencies, hasSize(3));
      assertDependency(bundleDependencies, 0, "zt-exec", empty(), TEST);
      assertDependency(bundleDependencies, 1, "commons-io", of("1.2"), TEST);
      assertDependency(bundleDependencies, 2, "slf4j-api", empty(), TEST);
    });
  }

  @Test
  public void scopeManaged() throws IOException {
    doTestWith("pom-manage-dependency-scope-managed/pom.xml", bundleDependencies -> {
      assertThat(bundleDependencies, hasSize(3));
      assertDependency(bundleDependencies, 0, "zt-exec", empty(), TEST);
      assertDependency(bundleDependencies, 1, "commons-io", of("1.4"), COMPILE);
      assertDependency(bundleDependencies, 2, "slf4j-api", empty(), TEST);
    });
  }

  private void assertDependency(List<BundleDependency> bundleDependencies, int position, String artifactId,
                                Optional<String> versionOptional, BundleScope scope) {
    BundleDependency bundleDependency = bundleDependencies.get(position);
    BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
    assertThat(bundleDescriptor.getArtifactId(), equalTo(artifactId));
    versionOptional.ifPresent(version -> assertThat(bundleDescriptor.getVersion(), equalTo(version)));
    assertThat(bundleDependency.getScope(), equalTo(scope));
    assertThat(bundleDependency.getBundleUri(), not(nullValue()));
  }

  private void doTestWith(String pomResourceFile, Consumer<List<BundleDependency>> task) throws IOException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(temporaryFolder.newFolder()));
    File artifactFile = toFile(getClass().getClassLoader().getResource(pomResourceFile));
    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(artifactFile, true, false, empty(), empty());
    task.accept(bundleDependencies);
  }

}
