/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.toFile;
import static org.codehaus.plexus.util.FileUtils.copyDirectoryStructure;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.RemoteRepositories.IGNORE_ARTIFACT_DESCRIPTOR_REPOSITORIES;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.eclipse.aether.resolution.DependencyResolutionException;
import org.junit.Test;

@Story(IGNORE_ARTIFACT_DESCRIPTOR_REPOSITORIES)
public class IgnoreRemoteRepositoriesFromArtifactTestCase extends AbstractMavenClientTestCase {

  private static final String MULE_CORE_EE_GAV = "com.mulesoft.mule:mule-core-ee:jar:3.3.0";
  private static final String MULE_CORE_GAV = "org.mule:mule-core:jar:3.3.0";

  @Test
  @Description("Validates that a pom with dependencies and remote repositories can be resolved if no settings is not provided (therefore using remote repositories from pom.xml file)")
  public void pomWithRemoteRepositoriesNoRemoteRepositoriesOnSettings() throws Exception {
    List<BundleDependency> dependencies = resolveDependencies("pom-with-repositories-no-dependencies", null);
    assertThat(dependencies, hasSize(1));
    assertThat(dependencies.get(0).getDescriptor().getArtifactId(), equalTo("redshift-jdbc42"));
  }

  @Test
  public void implicitMavenCentralFromPomFileShouldNotBeConsidered() throws Exception {
    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage("commons-collections:commons-collections:jar:3.2.2");
    resolveDependencies("pom-with-dependencies-from-maven-central", null, true);
  }

  @Test
  @Description("Validates that if no settings is provided, an empty(authenticationSelector) would be set and handled correctly but dependencies could not be fetched")
  public void failsToCollectDependenciesUsingPomWithSecureRepositoriesButNoSettingsIsProvided() throws Exception {
    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage(MULE_CORE_EE_GAV);
    resolveDependencies("pom-with-repositories-with-dependencies", null);
  }

  @Test
  @Description("Validates that if a settings is provided with correct serverIds but invalid credentials an authenticationSelector would be set and handled correctly but dependencies could not be fetched")
  public void failsToCollectDependenciesUsingPomWithSecureRepositoriesButSettingsIsProvidedWithInvalidCredentialsWhenResolvingArtifact()
      throws Exception {
    final String artifactId = "pom-with-repositories-with-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/settings.xml"));

    final File localMavenRepository = repositoryFolder.newFolder();
    copyDirectoryStructure(toFile(getClass().getClassLoader().getResource(artifactId + "/repository")),
                           localMavenRepository);

    final MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository);
    mavenConfigurationBuilder.userSettingsLocation(settings);
    mavenConfigurationBuilder.ignoreArtifactDescriptorRepositories(false);
    MavenConfiguration mavenConfiguration = mavenConfigurationBuilder.build();
    MavenClient mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    File pomFile = toFile(mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId("org.mule.tests")
        .setArtifactId(artifactId)
        .setVersion("1.0.0-SNAPSHOT")
        .setType("pom")
        .build()).getBundleUri().toURL());

    expectedException.expect(RuntimeException.class);
    expectedException.expectCause(allOf(
                                        instanceOf(DependencyResolutionException.class),
                                        hasProperty("message", containsString(MULE_CORE_EE_GAV))));
    mavenClient.resolveArtifactDependencies(pomFile, false, false, empty(), empty());
  }

  @Test
  @Description("Validates that if a settings is provided with wrong serverIds and the correct one but invalid credentials, an authenticationSelector would be set and handled correctly but dependencies could not be fetched")
  public void failsToCollectDependenciesUsingPomWithSecureRepositoriesButSettingsIsProvidedWithWrongServerIdsAndCredentials()
      throws Exception {
    final String artifactId = "pom-with-repositories-with-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/settings.xml"));

    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage(MULE_CORE_EE_GAV);
    resolveDependencies(artifactId, settings);
  }

  @Test
  @Description("Validates that if en empty settings is provided, an authenticationSelector (with no mappings) would be set and handled correctly and dependencies could be fetched from a public repository")
  public void resolvesDependencyUsingPomWithPublicRepositoriesAndEmptySettings() throws Exception {
    List<BundleDependency> dependencies = resolveDependencies("pom-with-repositories-with-public-dependencies", null);
    assertThat(dependencies, hasSize(1));
    assertCoreDependency(dependencies.get(0));
  }

  @Test
  @Description("Validates that remote repositories declared to the MavenConfiguration have precedence over the ones on pom, therefore artifacts cannot be found")
  public void failsToCollectDependenciesUsingPomWithRepositoriesOverridenOnMavenConfiguration() throws Exception {
    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage("com.amazon.redshift:redshift-jdbc42:jar:1.2.10.1009");
    resolveDependencies("pom-with-repositories-no-dependencies", null,
                        new RemoteRepository.RemoteRepositoryBuilder()
                            .id("redshift")
                            .url(new URL("http://redshift-maven-repository.s3-website-us-east-1.amazonaws.com/release-wrong"))
                            .build());
  }

  @Test
  @Description("Validates that Maven central repository from MavenConfiguration is ignored when ignorePomRepositories is disabled")
  public void mavenCentralIsIgnoredFromMavenConfigurationWhenPomRepositoriesAreNotIgnored() throws Exception {
    List<BundleDependency> dependencies = resolveDependencies("pom-with-dependency-available-in-central", null,
                                                              new RemoteRepository.RemoteRepositoryBuilder()
                                                                  .id("central")
                                                                  .url(new URL("https://repo.maven.apache.org/maven2-wrong"))
                                                                  .build());
    assertThat(dependencies, hasSize(1));
    assertThat(dependencies.get(0).getDescriptor().getArtifactId(), equalTo("commons-collections"));
  }

  @Test
  @Description("Validates that remote repositories declared on settings have precedence over the ones on pom, therefore artifacts cannot be found")
  public void failsToCollectDependenciesUsingPomWithRepositoriesOverridenBySettings() throws Exception {
    final String artifactId = "pom-with-repositories-no-dependencies";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/wrong-server-urls-settings.xml"));

    expectedException.expect(BundleDependenciesResolutionException.class);
    expectedException.expectMessage("com.amazon.redshift:redshift-jdbc42:jar:1.2.10.1009");
    resolveDependencies(artifactId, settings);
  }

  private void assertCoreDependency(BundleDependency coreDependency) {
    assertThat(coreDependency.getDescriptor().getGroupId(), is("org.mule"));
    assertThat(coreDependency.getDescriptor().getArtifactId(), is("mule-core"));
    assertThat(coreDependency.getDescriptor().getVersion(), is("3.3.0"));
    assertThat(coreDependency.getDescriptor().getClassifier().isPresent(), is(false));
  }

  private List<BundleDependency> resolveDependencies(String artifactId, File settings,
                                                     boolean ignoreArtifactDescriptorRepositories,
                                                     RemoteRepository... remoteRepositories)
      throws IOException {
    final File localMavenRepository = repositoryFolder.newFolder();
    copyDirectoryStructure(toFile(getClass().getClassLoader().getResource(artifactId + "/repository")),
                           localMavenRepository);

    final MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository);
    if (settings != null) {
      mavenConfigurationBuilder.userSettingsLocation(settings);
    }

    for (RemoteRepository remoteRepository : remoteRepositories) {
      mavenConfigurationBuilder.remoteRepository(remoteRepository);
    }
    mavenConfigurationBuilder.ignoreArtifactDescriptorRepositories(ignoreArtifactDescriptorRepositories);

    MavenConfiguration mavenConfiguration = mavenConfigurationBuilder.build();
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    final BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.tests")
        .setArtifactId(artifactId)
        .setVersion("1.0.0-SNAPSHOT")
        .setType("pom")
        .build();

    return getBundleDependencies(bundleDescriptor);
  }

  private List<BundleDependency> resolveDependencies(String artifactId, File settings, RemoteRepository... remoteRepositories)
      throws IOException {
    return resolveDependencies(artifactId, settings, false, remoteRepositories);
  }

  private List<BundleDependency> getBundleDependencies(BundleDescriptor bundleDescriptor) {
    BundleDependency pomBundle = mavenClient.resolveBundleDescriptor(bundleDescriptor);
    assertThat(pomBundle.getDescriptor().getGroupId(), is(bundleDescriptor.getGroupId()));
    assertThat(pomBundle.getDescriptor().getArtifactId(), is(bundleDescriptor.getArtifactId()));
    assertThat(pomBundle.getDescriptor().getVersion(), is(bundleDescriptor.getVersion()));
    assertThat(pomBundle.getDescriptor().getType(), is(bundleDescriptor.getType()));
    assertThat(pomBundle.getDescriptor().getClassifier().isPresent(), is(false));

    return mavenClient.resolveBundleDescriptorDependencies(false, bundleDescriptor);
  }

  @Override
  protected void beforeTest() throws Exception {}

}
