/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;

import java.io.File;
import java.io.IOException;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper class to work with tests using mule maven client
 *
 * @since 1.0
 */
public class MavenTestHelper {

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String MULESOFT_PRIVATE_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/private/";
  private static final String DRY_RUN_REPO_ID = "mule.test.dryRunNexusId";
  private static final String DRY_RUN_REPO_URL = "mule.test.dryRunNexusUrl";
  private static Logger LOGGER = LoggerFactory.getLogger(MavenTestHelper.class);

  /**
   * Creates a default maven configuration for enterprise edition.
   *
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultEnterpriseMavenConfiguration() throws IOException {
    return createDefaultEnterpriseMavenConfiguration(false);
  }

  /**
   * Creates a default maven configuration for enterprise edition.
   *
   * @param forceUpdateSnapshots {@code true} if updates should be forced to be updated.
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultEnterpriseMavenConfiguration(boolean forceUpdateSnapshots) throws IOException {
    MavenConfiguration mavenConfiguration = createDefaultEnterpriseMavenConfigurationBuilder(forceUpdateSnapshots).build();
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Using MavenConfiguration {}", mavenConfiguration);
    }
    return mavenConfiguration;
  }

  /**
   * Creates a default maven configuration for community edition.
   *
   * @param forceUpdateSnapshots {@code true} if updates should be forced to be updated.
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultCommunityMavenConfiguration(boolean forceUpdateSnapshots) throws IOException {
    MavenConfiguration mavenConfiguration = createDefaultCommunityMavenConfigurationBuilder(forceUpdateSnapshots).build();
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Using MavenConfiguration {}", mavenConfiguration);
    }
    return mavenConfiguration;
  }

  /**
   * Creates a default maven configuration for community edition.
   *
   * @return a new maven configuration based on the user environment.
   */
  public static MavenConfiguration createDefaultCommunityMavenConfiguration() throws IOException {
    return createDefaultCommunityMavenConfiguration(false);
  }

  /**
   * Creates a maven config builder with the enterprise default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultEnterpriseMavenConfigurationBuilder()
      throws IOException {
    return createDefaultMavenConfigurationBuilder(true, false);
  }

  /**
   * Creates a maven config builder with the enterprise default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultEnterpriseMavenConfigurationBuilder(boolean forceUpdateSnapshots)
      throws IOException {
    return createDefaultMavenConfigurationBuilder(true, forceUpdateSnapshots);
  }

  /**
   * Creates a maven config builder with the community default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultCommunityMavenConfigurationBuilder()
      throws IOException {
    return createDefaultMavenConfigurationBuilder(false, false);
  }

  /**
   * Creates a maven config builder with the community default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfiguration.MavenConfigurationBuilder createDefaultCommunityMavenConfigurationBuilder(boolean forceUpdateSnapshots)
      throws IOException {
    return createDefaultMavenConfigurationBuilder(false, forceUpdateSnapshots);
  }

  /**
   * @return the local repository folder discovered for the environment.
   */
  public static File getLocalRepositoryFolder() {
    return new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();
  }

  /**
   * Creates a maven config builder with the default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  private static MavenConfiguration.MavenConfigurationBuilder createDefaultMavenConfigurationBuilder(boolean enterprise,
                                                                                                     boolean forceUpdateSnapshots)
      throws IOException {
    MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder =
        newMavenConfigurationBuilder()
            .forcePolicyUpdateNever(!forceUpdateSnapshots)
            .forcePolicyUpdateAlways(forceUpdateSnapshots);

    final File localMavenRepository = getLocalRepositoryFolder();
    mavenConfigurationBuilder.localMavenRepositoryLocation(localMavenRepository);

    final DefaultSettingsSupplierFactory settingsSupplierFactory =
        new DefaultSettingsSupplierFactory(new MavenEnvironmentVariables());

    mavenConfigurationBuilder.remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
        .id("mulesoft-public")
        .url(new URL(MULESOFT_PUBLIC_REPOSITORY))
        .build());
    if (enterprise) {
      mavenConfigurationBuilder.remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
          .id("mulesoft-private")
          .url(new URL(MULESOFT_PRIVATE_REPOSITORY))
          .build());
    }
    if (System.getProperty(DRY_RUN_REPO_URL) != null) {
      mavenConfigurationBuilder.remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
          .id(System.getProperty(DRY_RUN_REPO_ID, "mule-runtime-nexus-test-releases"))
          .url(new URL(System.getProperty(DRY_RUN_REPO_URL)))
          .build());
    }
    settingsSupplierFactory.environmentUserSettingsSupplier().ifPresent(mavenConfigurationBuilder::userSettingsLocation);
    settingsSupplierFactory.environmentGlobalSettingsSupplier().ifPresent(mavenConfigurationBuilder::globalSettingsLocation);
    settingsSupplierFactory.environmentSettingsSecuritySupplier().ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);
    return mavenConfigurationBuilder;
  }
}
