/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.aether.resolution.DependencyResolutionException;
import org.junit.Test;

public class ResolvePluginBundleDescriptorsDependenciesTestCase extends AbstractMavenClientTestCase {

  private List<BundleDescriptor> bundleDescriptors;

  @Override
  protected void beforeTest() throws Exception {
    bundleDescriptors = new ArrayList<>();
    BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId("org.mule.connectors")
        .setArtifactId("mule-http-connector")
        .setVersion("1.5.1")
        .setType("jar")
        .setClassifier("mule-plugin")
        .build();

    bundleDescriptors.add(bundleDescriptor);
  }

  @Test
  public void shouldResolvePluginBundleDescriptorsDependenciesUsingRemoteRepositoriesFromMavenConfiguration() throws IOException {
    RemoteRepository remoteRepository = new RemoteRepository.RemoteRepositoryBuilder()
        .id("mulesoft-public")
        .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public/"))
        .build();

    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.newFolder())
        .remoteRepository(remoteRepository)
        .build();
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    List<BundleDependency> bundleDependencies = mavenClient.resolvePluginBundleDescriptorsDependencies(bundleDescriptors);
    assertThat(bundleDependencies.size(), is(2));
  }

  @Test
  public void shouldFailToResolvePluginBundleDescriptorsDependenciesWhenNoRemoteRepositoriesAreDefinedInMavenConfiguration()
      throws IOException {
    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.newFolder())
        .build();
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    expectedException.expect(RuntimeException.class);
    expectedException.expectCause(allOf(
                                        instanceOf(DependencyResolutionException.class),
                                        hasProperty("message", containsString("mule-http-connector"))));
    mavenClient.resolvePluginBundleDescriptorsDependencies(bundleDescriptors);
  }

}
