/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.lang3.exception.ExceptionUtils.getRootCause;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.fail;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_ALWAYS;
import static org.mule.maven.client.api.model.RepositoryPolicy.UPDATE_POLICY_NEVER;
import static org.mule.maven.client.test.AllureConstants.MavenClient.RemoteRepositories.REPOSITORY_POLICY;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.api.model.RepositoryPolicy;
import org.mule.maven.test.ArtifactCreator;

import java.io.File;
import java.net.MalformedURLException;
import java.util.List;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(REPOSITORY_POLICY)
public class RepositoryReleasesPolicyTestCase extends AbstractMavenClientTestCase {

  @Override
  protected void beforeTest() {}

  private MavenConfiguration.MavenConfigurationBuilder createMavenConfigurationBuilder(RemoteRepository remoteRepository) {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.getRoot())
        .remoteRepository(remoteRepository);
  }

  @Test
  public void updatePolicyAlwaysDeclaredAtRepositoryLevel() {
    doTestUpdatePolicy(false, false, RepositoryPolicy.newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_ALWAYS)
        .build());
  }

  @Test
  public void updatePolicyNeverHasPrecedence() {
    doTestUpdatePolicy(true, false, RepositoryPolicy.newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_ALWAYS)
        .build());
  }

  @Test
  public void forceUpdateSnapshotsHasPrecedence() {
    doTestUpdatePolicy(false, true, RepositoryPolicy.newRepositoryPolicyBuilder()
        .enabled(true)
        .updatePolicy(UPDATE_POLICY_NEVER)
        .build());
  }

  private void doTestUpdatePolicy(boolean forceUpdatePolicyNever, boolean forceUpdateSnapshots,
                                  RepositoryPolicy repositoryPolicy) {
    RemoteRepository.RemoteRepositoryBuilder remoteRepositoryBuilder = newRemoteRepositoryBuilder();
    try {
      remoteRepositoryBuilder
          .id("3party-repository")
          .url(repositoryFolder.getRoot().toURI().toURL());
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }

    if (repositoryPolicy != null) {
      remoteRepositoryBuilder.releasePolicy(repositoryPolicy);
    }
    RemoteRepository remoteRepository = remoteRepositoryBuilder.build();

    File pomFile = toFile(this.getClass().getClassLoader().getResource("repository-releases-policy/pom.xml"));
    MavenConfiguration.MavenConfigurationBuilder mavenConfiguration = createMavenConfigurationBuilder(remoteRepository);
    mavenConfiguration.forcePolicyUpdateNever(forceUpdatePolicyNever);
    mavenConfiguration.forcePolicyUpdateAlways(forceUpdateSnapshots);
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration.build());

    try {
      mavenClient.resolveArtifactDependencies(pomFile, false, false,
                                              empty(),
                                              empty());
      fail();
    } catch (RuntimeException e) {
      assertThat(getRootCause(e).getMessage(),
                 containsString("Could not find artifact org.mule.test:artifact:pom:1.0 in 3party-repository"));
    }

    // Install the dependency in the remote repository
    repositoryFolder.addArtifacts(ArtifactCreator.artifact("org.mule.test", "artifact", "1.0").build());
    try {
      List<BundleDependency> bundleDependencies = mavenClient.resolveArtifactDependencies(pomFile, false, false,
                                                                                          empty(),
                                                                                          empty());
      assertThat(forceUpdatePolicyNever, is(false));

      assertThat(bundleDependencies, hasSize(1));
      BundleDescriptor descriptor = bundleDependencies.get(0).getDescriptor();
      assertThat(descriptor.getGroupId(), is("org.mule.test"));
      assertThat(descriptor.getArtifactId(), is("artifact"));
      assertThat(descriptor.getVersion(), is("1.0"));
    } catch (RuntimeException e) {
      assertThat(forceUpdatePolicyNever, is(true));
      String message = getRootCause(e).getMessage();
      assertThat(message, startsWith("Failure to find org.mule.test:artifact:pom:1.0"));
      assertThat(message,
                 endsWith("was cached in the local repository, resolution will not be reattempted until the update interval of 3party-repository has elapsed or updates are forced"));
    }
  }

}
