/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.VersionRangeStory.RESOLVE_VERSION_RANGE_STORY;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;

import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;

import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.aether.util.version.GenericVersionScheme;

import org.junit.Test;

import io.qameta.allure.Story;

// TODO W-12334992: Use a mock Maven repository instead of an actual Nexus.
@Story(RESOLVE_VERSION_RANGE_STORY)
public class VersionRangeTestCase extends AbstractMavenClientTestCase {

  private final String groupId = "org.mule";
  private final String artifactId = "mule-core";
  private final String version = "[3.5.0,3.6.0)";
  private final String type = "jar";

  @Override
  protected void beforeTest() throws Exception {}

  private MavenConfiguration createMavenConfiguration()
      throws MalformedURLException {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.getRoot())
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder()
            .id("mulesoft-public")
            .url(new URL("https://repository-master.mulesoft.org/nexus/content/repositories/public/"))
            .build())
        .build();
  }

  @Test
  public void resolveRange() throws MalformedURLException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());

    final VersionRangeResult versionRangeResult =
        mavenClient.resolveVersionRange(new BundleDescriptor.Builder().setGroupId(groupId)
            .setArtifactId(artifactId)
            .setVersion(version)
            .setType(type).build());

    assertThat(versionRangeResult.getVersions().toString(),
               versionRangeResult.getVersions(), hasSize(greaterThanOrEqualTo(2)));
    assertThat(versionRangeResult.getLowestVersion(), equalTo("3.5.0"));
    assertThat(versionRangeResult.getHighestVersion(), equalTo("3.5.5-SNAPSHOT"));
  }

  @Test
  public void resolveSimpleVersion() throws MalformedURLException {
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration());

    final VersionRangeResult versionRangeResult =
        mavenClient.resolveVersionRange(new BundleDescriptor.Builder().setGroupId(groupId)
            .setArtifactId(artifactId)
            .setVersion("3.3.0")
            .setType(type).build());

    assertThat(versionRangeResult.getVersions().toString(),
               versionRangeResult.getVersions(), hasSize(1));
    assertThat(versionRangeResult.getLowestVersion(), equalTo("3.3.0"));
    assertThat(versionRangeResult.getHighestVersion(), equalTo("3.3.0"));
  }

  @Test
  public void resolveRangeVersionWithBuildNumber() throws Exception {
    // Code from Aether internal implementation as there is no available GAVs with the conditions to test this on central
    // Maven versioning format is: <MajorVersion [> . <MinorVersion [> . <IncrementalVersion ] ] [> - <BuildNumber | Qualifier ]>
    final GenericVersionScheme genericVersionScheme = new GenericVersionScheme();
    assertThat(genericVersionScheme.parseVersion("0.9.v20140226").compareTo(genericVersionScheme.parseVersion("0.10.v20140226")),
               is(-1));
    assertThat(genericVersionScheme.parseVersion("0.9.0.v20140226")
        .compareTo(genericVersionScheme.parseVersion("0.10.0.v20140226")), is(-1));
    assertThat(genericVersionScheme.parseVersion("1.1.0").compareTo(genericVersionScheme.parseVersion("1.1.0-1")), is(-1));
    assertThat(genericVersionScheme.parseVersion("1.1.0-1").compareTo(genericVersionScheme.parseVersion("1.1.0-2")), is(-1));
    assertThat(genericVersionScheme.parseVersion("1.1.0-2").compareTo(genericVersionScheme.parseVersion("1.1.1")), is(-1));
    assertThat(genericVersionScheme.parseVersion("1.1.1").compareTo(genericVersionScheme.parseVersion("1.1.11")), is(-1));
  }

}
