/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.ProxyStory.PROXY_STORY;

import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.Matchers.both;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;

import org.mule.maven.client.api.BundleDependenciesResolutionException;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;

import io.qameta.allure.Story;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.junit.Test;

@Story(PROXY_STORY)
public class ProxyTestCase extends AbstractMavenClientTestCase {

  private String groupId = "org.mule";
  private String artifactId = "mule-core";
  private String version = "3.3.0";
  private String type = "pom";

  @Override
  protected void beforeTest() {}

  private MavenConfiguration createMavenConfiguration(File userSettingsLocation, RemoteRepository remoteRepository) {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(repositoryFolder.getRoot())
        .remoteRepository(remoteRepository)
        .userSettingsLocation(userSettingsLocation)
        .build();
  }

  @Test
  public void useProxy() throws MalformedURLException {
    final RemoteRepository remoteRepository = newRemoteRepositoryBuilder()
        .id("mulesoft-public")
        .url(new URL("https://repository.mulesoft.org/nexus/content/repositories/public/"))
        .build();

    URL settingsUrl = getClass().getClassLoader().getResource("settings-with-proxy.xml");
    mavenClient = mavenClientProvider.createMavenClient(createMavenConfiguration(toFile(settingsUrl), remoteRepository));

    try {
      mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder().setGroupId(groupId)
          .setArtifactId(artifactId)
          .setVersion(version)
          .setType(type).build());
      fail();
    } catch (BundleDependenciesResolutionException e) {
      assertThat(e.getMessage(), both(containsString("Could not transfer artifact org.mule:mule-core:pom:3.3.0"))
          .and(containsString("proxy.testing.com")));
      final Throwable rootCause = ExceptionUtils.getRootCause(e);
      assertThat(rootCause, instanceOf(UnknownHostException.class));
    }
  }

}
