/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static java.lang.System.getProperty;
import static org.mule.maven.client.internal.util.FileUtils.loadFileContentFrom;
import static org.mule.maven.pom.parser.internal.util.FileUtils.getPomUrlFromJar;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.function.Supplier;

import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

/**
 * Provide a set of utility methods to query data from the {@code pom.xml} file of a test's project.
 *
 * @since 1.0
 */
public final class MavenTestUtils {

  private static final String POM_XML = "pom.xml";
  private static final String JAR = ".jar";

  private MavenTestUtils() {
    // Nothing to do
  }

  /**
   * Returns the value of the property defined or inherited in the {@code pom.xml} in the folder pointed by
   * {@code pomFolderFinder}.
   */
  public static String getMavenProperty(String property, Supplier<File> pomFolderFinder) {
    String propertyValue = getProperty(property);
    if (propertyValue != null) {
      return propertyValue;
    }

    Model mavenProject = mavenProject(pomFolderFinder);
    propertyValue = (String) mavenProject.getProperties().get(property);

    return resolveMavenProperty(propertyValue, mavenProject);
  }

  /**
   * Returns the version of the project in the {@code pom.xml} in the folder pointed by {@code pomFolderFinder}.
   */
  public static String getMavenProjectVersion(Supplier<File> pomFolderFinder) {
    Model mavenProject = mavenProject(pomFolderFinder);
    String muleVersion = resolveMavenProperty(mavenProject.getVersion(), mavenProject);

    if (muleVersion != null) {
      return muleVersion;
    }

    return mavenProject.getParent().getVersion();
  }

  /**
   * Finds the Maven's pom file corresponding to a given class that is part of a Maven project
   *
   * @param clazz class used to locate the pom for the Maven project that the class belongs to
   * @return a supplier that will provide the required file
   */
  public static Supplier<File> mavenPomFinder(Class clazz) {
    return () -> {
      try {
        return new File(clazz.getProtectionDomain().getCodeSource().getLocation()
            .toURI()).getParentFile().getParentFile().getParentFile();
      } catch (URISyntaxException e) {
        throw new IllegalArgumentException(e);
      }
    };
  }

  /**
   * Builds a Maven project representation for the {@code pom.xml} in the folder pointed by {@code pomFolderFinder}.
   */
  private static Model mavenProject(Supplier<File> pomFolderFinder) {
    return createMavenProject(pomFolderFinder.get());
  }

  /**
   * If {@code propertyValue} is a placeholder, it will attempt top be resolved within the provided {@code mavenProject}
   */
  private static String resolveMavenProperty(String propertyValue, Model mavenProject) {
    if (propertyValue != null && propertyValue.startsWith("${") && propertyValue.endsWith("}")) {
      String property = propertyValue.substring(propertyValue.indexOf("{") + 1, propertyValue.lastIndexOf("}"));
      return mavenProject.getProperties().get(property).toString();
    } else {
      return propertyValue;
    }
  }

  private static Model createMavenProject(File pomFile) {
    if (pomFile != null && pomFile.exists()) {
      // read pom file from Maven metadata inside the jar
      if (pomFile.isFile() && pomFile.getName().endsWith(JAR)) {
        try {
          MavenXpp3Reader reader = new MavenXpp3Reader();
          return reader.read(new ByteArrayInputStream(loadFileContentFrom(getPomUrlFromJar(pomFile)).get()));
        } catch (IOException | XmlPullParserException e) {
          throw new RuntimeException(e);
        }
      }
      // assumes that this is a project root with the sources file therefore the pom.xml should be at the root level
      MavenXpp3Reader mavenReader = new MavenXpp3Reader();
      try (FileReader reader = new FileReader(new File(pomFile, POM_XML))) {
        Model model = mavenReader.read(reader);
        model.setPomFile(pomFile);

        return model;
      } catch (Exception e) {
        throw new RuntimeException("Couldn't get Maven Artifact from pom: " + pomFile, e);
      }
    }
    throw new IllegalArgumentException("pom file doesn't exits for path: " + pomFile);
  }

}
