/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;
import static org.mule.maven.client.test.AllureConstants.MavenClient.RemoteRepositories.REPOSITORY_POLICY;

import static java.util.Optional.empty;

import static org.apache.commons.io.FileUtils.copyDirectory;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.net.MalformedURLException;
import java.util.List;

import io.qameta.allure.Story;
import org.junit.Test;

@Story(REPOSITORY_POLICY)
public class RepositorySnapshotsPolicyTestCase extends AbstractMavenClientTestCase {

  @Override
  protected void beforeTest() {}

  private MavenConfiguration.MavenConfigurationBuilder createMavenConfigurationBuilder(RemoteRepository remoteRepository) {
    return newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(temporaryFolder.getRoot())
        .remoteRepository(remoteRepository);
  }

  @Test
  public void updatePolicyAlwaysShouldFetchNewSnapshot() throws Exception {
    RemoteRepository.RemoteRepositoryBuilder remoteRepositoryBuilder = newRemoteRepositoryBuilder();
    try {
      remoteRepositoryBuilder
          .id("3party-repository")
          .url(repositoryFolder.getRoot().toURI().toURL());
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }

    RemoteRepository remoteRepository = remoteRepositoryBuilder.build();

    File pomFile = toFile(this.getClass().getClassLoader().getResource("repository-snapshots-policy/pom.xml"));
    MavenConfiguration.MavenConfigurationBuilder mavenConfiguration = createMavenConfigurationBuilder(remoteRepository);
    mavenConfiguration.forcePolicyUpdateNever(false);
    mavenConfiguration.forcePolicyUpdateAlways(true);
    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration.build());

    // Copy mocked artifacts along with metadata that has the first version of the snapshot with dependencies
    copyDirectory(toFile(this.getClass().getClassLoader().getResource("repository-snapshots-policy/repository/before")),
                  repositoryFolder.getRoot(), true);
    List<BundleDependency> bundleDependencies = mavenClient.resolveArtifactDependencies(pomFile, false, false,
                                                                                        empty(),
                                                                                        empty());
    assertThat(bundleDependencies, hasSize(1));
    BundleDependency bundleDependency = bundleDependencies.get(0);
    assertThat(bundleDependency.getTransitiveDependencies(), hasSize(0));
    BundleDescriptor descriptor = bundleDependency.getDescriptor();
    assertThat(descriptor.getGroupId(), is("org.mule.test"));
    assertThat(descriptor.getArtifactId(), is("artifact"));
    assertThat(descriptor.getBaseVersion(), is("1.0-SNAPSHOT"));
    assertThat(descriptor.getVersion(), is("1.0-20190218.204154-1"));

    // Copy mocked artifacts along with metadata that has a new version of the snapshot with a declared dependency
    copyDirectory(toFile(this.getClass().getClassLoader().getResource("repository-snapshots-policy/repository/after")),
                  repositoryFolder.getRoot(), true);
    bundleDependencies = mavenClient.resolveArtifactDependencies(pomFile, false, false,
                                                                 empty(),
                                                                 empty());
    assertThat(bundleDependencies, hasSize(2));
    bundleDependency = bundleDependencies.get(0);
    assertThat(bundleDependency.getTransitiveDependencies(), hasSize(1));
    assertThat(bundleDependency.getTransitiveDependencies().get(0).getDescriptor().getArtifactId(), is("artifactDependency"));
    descriptor = bundleDependency.getDescriptor();
    assertThat(descriptor.getGroupId(), is("org.mule.test"));
    assertThat(descriptor.getArtifactId(), is("artifact"));
    assertThat(descriptor.getBaseVersion(), is("1.0-SNAPSHOT"));
    assertThat(descriptor.getVersion(), is("1.0-20190219.202134-2"));
  }

}
