/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import static java.lang.String.valueOf;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.equalTo;
import static com.github.tomakehurst.wiremock.client.WireMock.exactly;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.getRequestedFor;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static com.github.tomakehurst.wiremock.client.WireMock.urlPathMatching;
import static com.github.tomakehurst.wiremock.client.WireMock.verify;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.http.HttpHeaders.AUTHORIZATION;
import static org.codehaus.plexus.util.FileUtils.copyDirectoryStructure;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import java.util.stream.Stream;

import com.github.tomakehurst.wiremock.http.Fault;
import com.github.tomakehurst.wiremock.junit.WireMockClassRule;
import com.github.tomakehurst.wiremock.stubbing.Scenario;

import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Issue;

public class WagonServerConfigurationTestCase extends AbstractMavenClientTestCase {

  @ClassRule
  public static WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig()
      .dynamicPort().bindAddress("localhost"));

  @Rule
  public WireMockClassRule wireMock = wireMockRule;

  private File pomFile;
  private String pomWithPort;

  @Override
  protected void beforeTest() throws IOException {
    final String artifactId = "pom-with-dependencies-settings-with-server";
    File settings = toFile(getClass().getClassLoader().getResource(artifactId + "/settings.xml"));

    final File localMavenRepository = repositoryFolder.newFolder();
    copyDirectoryStructure(requireNonNull(toFile(getClass().getClassLoader().getResource(artifactId + "/repository"))),
                           localMavenRepository);

    MavenConfiguration mavenConfiguration = newMavenConfigurationBuilder()
        .localMavenRepositoryLocation(localMavenRepository).userSettingsLocation(settings)
        .ignoreArtifactDescriptorRepositories(false).build();

    mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);
    pomFile = toFile(mavenClient.resolveBundleDescriptor(new BundleDescriptor.Builder()
        .setGroupId("org.mule.tests")
        .setArtifactId(artifactId)
        .setVersion("1.0.0-SNAPSHOT")
        .setType("pom")
        .build()).getBundleUri().toURL());
    pomWithPort = readFile(pomFile.getPath()).replace("dynamicPort", valueOf(wireMock.port()));
    Files.write(pomFile.toPath(), pomWithPort.getBytes());
  }

  @Test
  @Issue("Maven client support of http configuration on servers")
  public void downloadDependenciesWithServerUsingHttpHeaders() {
    wireMock.stubFor(
                     get(urlPathMatching("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom"))
                         .withHeader(AUTHORIZATION, equalTo("ThisIsASecret"))
                         .willReturn(aResponse()
                             .withBody(pomWithPort.getBytes())
                             .withHeader("Content-Type", "application/json")
                             .withStatus(200)));

    wireMock.stubFor(
                     get(urlPathMatching("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom.sha1"))
                         .withHeader(AUTHORIZATION, equalTo("ThisIsASecret"))
                         .willReturn(aResponse()
                             .withBody("0d4f2e4347623db8fbc76ac57236a325039a25ef")
                             .withHeader("Content-Type", "plain/text")
                             .withStatus(200)));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(pomFile, false, false, empty(), empty());

    // 1) 200, 2) but as the checksum fails Aether does another retry
    verify(exactly(2), getRequestedFor(urlEqualTo("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom")));
    verify(exactly(2),
           getRequestedFor(urlEqualTo("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom.sha1")));

    verifyDependency(bundleDependencies);
  }

  /**
   * Test to validate retries when there is an error while connecting to a remote repository. From Maven Wagon documentation:
   * {code} By default it would only retry if the is an exception. "Any retry handler can only react to exceptions when executing
   * the request and receiving the response head. It will not salvage in-flight failures of ongoing response body streams."
   * https://maven.apache.org/wagon/wagon-providers/wagon-http/ {code}
   *
   * As an alternative they also provide another system property but it is initialized statically:
   * "maven.wagon.http.serviceUnavailableRetryStrategy.class", "standard"
   *
   * When this property is set it would also retry if the HTTP response is: {code} statusCode == 408 || statusCode == 429 ||
   * statusCode == 500 || statusCode == 502 || statusCode == 503 || statusCode == 504; {code}
   *
   */
  @Test
  @Ignore("W-17369306")
  public void httpRequestRetryHandler() {
    wireMock.stubFor(
                     get(urlPathMatching("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom"))
                         .withHeader(AUTHORIZATION, equalTo("ThisIsASecret"))
                         .inScenario("Retry HTTP Scenario")
                         .whenScenarioStateIs(Scenario.STARTED)
                         .willReturn(aResponse().withFault(Fault.EMPTY_RESPONSE))
                         .willSetStateTo("FirstRetry"));

    wireMock.stubFor(
                     get(urlPathMatching("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom"))
                         .withHeader(AUTHORIZATION, equalTo("ThisIsASecret"))
                         .inScenario("Retry HTTP Scenario")
                         .whenScenarioStateIs("FirstRetry")
                         .willReturn(aResponse()
                             .withBody(pomWithPort.getBytes())
                             .withHeader("Content-Type", "application/json")
                             .withStatus(200)));

    wireMock.stubFor(
                     get(urlPathMatching("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom.sha1"))
                         .withHeader(AUTHORIZATION, equalTo("ThisIsASecret"))
                         .willReturn(aResponse()
                             .withBody("0d4f2e4347623db8fbc76ac57236a325039a25ef")
                             .withHeader("Content-Type", "plain/text")
                             .withStatus(200)));

    List<BundleDependency> bundleDependencies =
        mavenClient.resolveArtifactDependencies(pomFile, false, false, empty(), empty());

    // Verify GET request was made again after first attempt
    // 1) Would throw CONNECTION_RESET_BY_PEER, 2) 200, 3) but as the checksum fails Aether does another retry
    verify(exactly(3), getRequestedFor(urlEqualTo("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom")));
    verify(exactly(2),
           getRequestedFor(urlEqualTo("/com/mulesoft/mule/tests/mule-tests-core/4.2.1/mule-tests-core-4.2.1.pom.sha1")));

    verifyDependency(bundleDependencies);
  }

  private String readFile(String filePath) throws IOException {
    StringBuilder stringBuilder = new StringBuilder();
    Stream<String> stream = Files.lines(Paths.get(filePath));
    stream.forEach(s -> stringBuilder.append(s).append("\n"));
    return stringBuilder.toString();
  }

  private void verifyDependency(List<BundleDependency> bundleDependencies) {
    assertThat(bundleDependencies, hasSize(1));
    BundleDependency dependency = bundleDependencies.get(0);
    BundleDescriptor dependencyDescriptor = dependency.getDescriptor();
    assertThat(dependencyDescriptor.getGroupId(), is("com.mulesoft.mule.tests"));
    assertThat(dependencyDescriptor.getArtifactId(), is("mule-tests-core"));
    assertThat(dependencyDescriptor.getVersion(), is("4.2.1"));
    assertThat(dependencyDescriptor.getBaseVersion(), is("4.2.1"));
    assertThat(dependencyDescriptor.getType(), is("pom"));
  }
}
