/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.test;

import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_FAIL;
import static org.mule.maven.client.api.model.RepositoryPolicy.CHECKSUM_POLICY_WARN;
import static org.mule.maven.client.internal.DefaultSettingsSupplierFactory.GLOBAL_SETTINGS_SYSTEM_PROPERTY;
import static org.mule.maven.client.internal.DefaultSettingsSupplierFactory.MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY;
import static org.mule.maven.client.internal.DefaultSettingsSupplierFactory.USER_SETTINGS_SYSTEM_PROPERTY;
import static org.mule.maven.client.test.AllureConstants.MavenClient.MAVEN_CLIENT;
import static org.mule.maven.client.test.AllureConstants.MavenClient.SettingsSupplierFactoryStory.SETTINGS_SUPPLIER_FACTORY_STORY;

import static java.io.File.createTempFile;
import static java.lang.String.format;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Optional.empty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.maven.client.api.model.MavenConfiguration.MavenConfigurationBuilder;
import org.mule.maven.client.internal.DefaultSettingsSupplierFactory;
import org.mule.maven.client.internal.MavenEnvironmentVariables;

import java.io.File;
import java.io.IOException;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@Feature(MAVEN_CLIENT)
@Story(SETTINGS_SUPPLIER_FACTORY_STORY)
@RunWith(MockitoJUnitRunner.class)
public class DefaultSettingsSupplierFactoryTestCase {

  private static final String GS_S_COMMAND_LINE =
      "mvn -f pom.xml -s %s -gs %s  -Dbranch_wsc=main -Dbranch=main -Dextra_goals= -Dmaven.repo.local=/scratch/jenkins/workspace/.repository test -T 2 -Djava.net.preferIPv4Stack=true -fn -surefire.rerunFailingTestsCount=0";
  private static final String LONG_GS_S_COMMAND_LINE =
      "mvn -f pom.xml --settings %s --global-settings %s -Dbranch_wsc=main -Dbranch=main -Dextra_goals= -Dmaven.repo.local=/scratch/jenkins/workspace/.repository test -T 2 -Djava.net.preferIPv4Stack=true -fn -surefire.rerunFailingTestsCount=0";

  private File userSettings;
  private File globalSettings;
  private File settingsSecurity;
  private File userHomeDir;
  private File m2Dir;
  private File m2UserSettings;
  private File m2GlobalSettings;
  private File m2SettingsSecurity;

  @Mock
  private MavenEnvironmentVariables mavenEnvironmentVariables;

  @Before
  public void before() throws IOException {
    userSettings = createTempFile("settings", ".xml");
    globalSettings = createTempFile("globalSettings", ".xml");
    settingsSecurity = createTempFile("settings-security", ".xml");

    // Default structure for fallback cases to user folder
    userHomeDir = createTempDirectory("userHome").toFile();
    m2Dir = new File(userHomeDir, ".m2");
    m2Dir.mkdir();
    m2UserSettings = new File(m2Dir, "settings.xml");
    m2UserSettings.createNewFile();
    File m2ConfDir = new File(m2Dir, "conf");
    m2ConfDir.mkdir();
    m2GlobalSettings = new File(m2ConfDir, "settings.xml");
    m2GlobalSettings.createNewFile();
    m2SettingsSecurity = new File(m2Dir, "settings-security.xml");
    m2SettingsSecurity.createNewFile();
  }

  @Test
  public void duplicatedUserSettingsEntriesFirstShouldBeTaken() {
    String commandLine = format("--settings %s --settings %s", userSettings.getAbsolutePath(), globalSettings.getAbsolutePath());
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(commandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(userSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  @Test
  public void duplicatedGlobalSettingsEntriesFirstShouldBeTaken() {
    String commandLine =
        format("--global-settings %s --global-settings %s", globalSettings.getAbsolutePath(), userSettings.getAbsolutePath());
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(commandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(globalSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  @Test
  public void settingsSecurityFromMavenSystemProperty() {
    when(mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY))
        .thenReturn(settingsSecurity);

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentSettingsSecuritySupplier().get().getAbsolutePath(),
               is(settingsSecurity.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getFileAsSystemOrEnvProperty(MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY);
  }

  @Test
  public void settingsFromM2() {
    when(mavenEnvironmentVariables.getUserHome()).thenReturn(userHomeDir.getAbsolutePath());
    when(mavenEnvironmentVariables.getM2HomeEnv()).thenReturn(m2Dir.getAbsolutePath());

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(m2UserSettings.getAbsolutePath()));
    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(m2GlobalSettings.getAbsolutePath()));
    assertThat(settingsSupplierFactory.environmentSettingsSecuritySupplier().get().getAbsolutePath(),
               is(m2SettingsSecurity.getAbsolutePath()));
  }

  @Test
  public void settingsEmpty() {
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);
    settingsSupplierFactory.addToMavenConfig(mock(MavenConfigurationBuilder.class));

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier(), is(empty()));
    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier(), is(empty()));
    assertThat(settingsSupplierFactory.environmentSettingsSecuritySupplier(), is(empty()));
  }

  @Test
  public void globalSettingsShortCmdParam() {
    assertCommandLineGlobalSettings(GS_S_COMMAND_LINE);
  }

  @Test
  public void globalSettingsLongCmdParam() {
    assertCommandLineGlobalSettings(LONG_GS_S_COMMAND_LINE);
  }

  @Test
  public void userSettingsShortCmdParam() {
    assertCommandLineSettings(GS_S_COMMAND_LINE);
  }

  @Test
  public void userSettingsLongCmdParam() {
    assertCommandLineSettings(LONG_GS_S_COMMAND_LINE);
  }

  @Test
  public void globalSettingsAsSystemProperty() throws Exception {
    assertSystemPropertyGlobalSettings(globalSettings);
  }

  @Test
  public void userSettingsAsSystemProperty() throws Exception {
    assertUserSystemPropertySettings(userSettings);
  }

  @Test
  public void addToMavenConfigDefaultChecksumPolicy() {
    DefaultSettingsSupplierFactory settingsSupplierFactory = settingsSupplierWithAdditionalCmdLineArgs("");

    MavenConfigurationBuilder configurationBuilder = mock(MavenConfigurationBuilder.class);
    settingsSupplierFactory.addToMavenConfig(configurationBuilder);
    verify(configurationBuilder).userSettingsLocation(userSettings);
    verify(configurationBuilder).globalSettingsLocation(globalSettings);
    verify(configurationBuilder).settingsSecurityLocation(settingsSecurity);
    verify(configurationBuilder, times(0)).globalChecksumPolicy(any());
  }

  @Test
  public void addToMavenConfigStrictChecksum() {
    DefaultSettingsSupplierFactory settingsSupplierFactory = settingsSupplierWithAdditionalCmdLineArgs("--strict-checksums");

    MavenConfigurationBuilder configurationBuilder = mock(MavenConfigurationBuilder.class);
    settingsSupplierFactory.addToMavenConfig(configurationBuilder);
    verify(configurationBuilder).userSettingsLocation(userSettings);
    verify(configurationBuilder).globalSettingsLocation(globalSettings);
    verify(configurationBuilder).settingsSecurityLocation(settingsSecurity);
    verify(configurationBuilder).globalChecksumPolicy(CHECKSUM_POLICY_FAIL);
  }

  @Test
  public void addToMavenConfigLaxChecksum() {
    DefaultSettingsSupplierFactory settingsSupplierFactory = settingsSupplierWithAdditionalCmdLineArgs("--lax-checksums");

    MavenConfigurationBuilder configurationBuilder = mock(MavenConfigurationBuilder.class);
    settingsSupplierFactory.addToMavenConfig(configurationBuilder);
    verify(configurationBuilder).userSettingsLocation(userSettings);
    verify(configurationBuilder).globalSettingsLocation(globalSettings);
    verify(configurationBuilder).settingsSecurityLocation(settingsSecurity);
    verify(configurationBuilder).globalChecksumPolicy(CHECKSUM_POLICY_WARN);
  }

  private void assertCommandLineGlobalSettings(String commandLine) {
    String formattedCommandLine = format(commandLine, userSettings.getAbsolutePath(), globalSettings.getAbsolutePath());

    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(formattedCommandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(globalSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  private void assertCommandLineSettings(String commandLine) {
    String formattedCommandLine = format(commandLine, userSettings.getAbsolutePath(), globalSettings.getAbsolutePath());

    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(formattedCommandLine);
    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(userSettings.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
  }

  private void assertSystemPropertyGlobalSettings(File settingsFile) throws Exception {
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn("");
    when(mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY))
        .thenReturn(settingsFile);

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentGlobalSettingsSupplier().get().getAbsolutePath(),
               is(settingsFile.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
    verify(mavenEnvironmentVariables).getFileAsSystemOrEnvProperty(GLOBAL_SETTINGS_SYSTEM_PROPERTY);
  }

  private void assertUserSystemPropertySettings(File settingsFile) throws Exception {
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn("");
    when(mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY)).thenReturn(settingsFile);

    DefaultSettingsSupplierFactory settingsSupplierFactory = new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);

    assertThat(settingsSupplierFactory.environmentUserSettingsSupplier().get().getAbsolutePath(),
               is(settingsFile.getAbsolutePath()));
    verify(mavenEnvironmentVariables).getMavenCmdLineArgsEnv();
    verify(mavenEnvironmentVariables).getFileAsSystemOrEnvProperty(USER_SETTINGS_SYSTEM_PROPERTY);
  }

  private DefaultSettingsSupplierFactory settingsSupplierWithAdditionalCmdLineArgs(String additionalCmdLineArgs) {
    String commandLine = format("--settings %s --global-settings %s %s",
                                userSettings.getAbsolutePath(),
                                globalSettings.getAbsolutePath(),
                                additionalCmdLineArgs);
    when(mavenEnvironmentVariables.getMavenCmdLineArgsEnv()).thenReturn(commandLine);
    when(mavenEnvironmentVariables.getFileAsSystemOrEnvProperty(MAVEN_SETTINGS_SECURITY_SYSTEM_PROPERTY))
        .thenReturn(settingsSecurity);

    return new DefaultSettingsSupplierFactory(mavenEnvironmentVariables);
  }

}
