/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api;

import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.SharedLibrary;

import java.util.List;
import java.util.Map;
import java.util.Properties;

/**
 * Maven parser API to work with maven artifacts.
 *
 * @since 2.0.0
 */
public interface MavenPomParser {

  /**
   * Gets source directory declared in the pom.
   *
   * @return the source directory
   */
  String getSourceDirectory();

  /**
   * Gets resource directories declared in the pom.
   *
   * @return the resource directories
   */
  List<String> getResourceDirectories();

  /**
   * Gets {@link MavenPomModel} of the current pom.
   *
   * @return the {@link MavenPomModel}
   */
  MavenPomModel getModel();

  /**
   * Gets shared libraries declared configured in the Mule plugin.
   *
   * @return the shared libraries
   *
   * @apiNote Only for compatibility. AdditionalPluginDependencies must be used instead.
   */
  List<SharedLibrary> getSharedLibraries();

  /**
   * Gets the declared dependencies in the pom. Each {@link BundleDependency} will not contain the transitive dependencies.
   *
   * @return a list of the {@link BundleDependency}
   */
  List<BundleDependency> getDependencies();

  /**
   * Gets additional plugin dependencies configured in the Mule plugin. It returns a {@link Map} with an
   * {@link ArtifactCoordinates} key representing an artifact. The values of the map is the {@link AdditionalPluginDependencies}
   * for each artifact.
   *
   * @return the pom additional plugin dependencies
   */
  Map<ArtifactCoordinates, AdditionalPluginDependencies> getPomAdditionalPluginDependenciesForArtifacts();

  /**
   * Gets properties declared in the pom.
   *
   * @return the properties
   */
  Properties getProperties();

  /**
   * Searches the Maven {@link org.apache.maven.model.Model} for the {@url https://maven.apache.org/plugins/maven-shade-plugin/}.
   * maven-shade-plugin would have to be configured in the pom directly and not inherited from a parent pom due to it is not
   * supported by this implementation.
   *
   * @return true if found.
   */
  boolean isMavenShadePluginConfigured();

}
