/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import java.nio.file.Path;
import java.util.Iterator;
import java.util.Optional;
import java.util.ServiceLoader;

/**
 * API to find the implementation for the {@link MavenModelBuilder}
 *
 * @since 2.0
 */
public interface MavenModelBuilderProvider {

  /**
   * Creates a new {@link MavenModelBuilder} with the given parameters
   *
   * @param groupId      the groupId of the Model
   * @param artifactId   the artifactId of the Model
   * @param version      the version of the Model
   * @param modelVersion the modelVersion of the Model
   * @param packaging    the packaging of the model
   * @return a {@link MavenModelBuilder}
   */
  MavenModelBuilder createMavenModelBuilder(String groupId, String artifactId, String version,
                                            Optional<String> modelVersion, Optional<String> packaging);

  /**
   * Creates a new {@link MavenModelBuilder} from a file. This file could be a folder containing a POM, a file pointing to a POM
   * or a JAR.
   *
   * @param path the {@link Path} of the pom.
   * @return a {@link MavenModelBuilder}
   */
  MavenModelBuilder createMavenModelBuilder(Path path);

  /**
   * Discovers an available {@link MavenModelBuilderProvider}.
   *
   * @return a new client configured with the provided maven configuration.
   */
  static MavenModelBuilderProvider discoverProvider() {
    ServiceLoader<MavenModelBuilderProvider> factories = ServiceLoader.load(MavenModelBuilderProvider.class,
                                                                            MavenModelBuilderProvider.class.getClassLoader());
    Iterator<MavenModelBuilderProvider> iterator = factories.iterator();
    if (!iterator.hasNext()) {
      throw new IllegalStateException(String.format("Could not find %s service implementation through SPI",
                                                    MavenModelBuilderProvider.class.getName()));
    }
    return iterator.next();
  }

}
