/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import java.util.HashMap;
import java.util.List;

import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.startsWith;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.params.provider.Arguments.arguments;

class BundleDescriptorTestCase {

  private static final String GROUP_ID = "groupId1";
  private static final String GROUP_ID_2 = "groupId2";
  private static final String ARTIFACT_ID = "artifactId1";
  private static final String ARTIFACT_ID_2 = "artifactId2";
  private static final String VERSION = "version1";
  private static final String VERSION_2 = "version2";

  private static BundleDescriptor descriptor;
  private static BundleDescriptor pluginDescriptor;
  private static BundleDescriptor propertiedDescriptor;
  private static BundleDescriptor otherGroup;
  private static BundleDescriptor otherArtifact;
  private static BundleDescriptor otherVersion;

  @BeforeAll
  static void setup() {
    descriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .build();
    pluginDescriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .setClassifier("mule-plugin")
        .build();
    HashMap<String, String> properties = new HashMap<>();
    properties.put("foo", "bar");
    propertiedDescriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .setProperties(properties)
        .build();
    otherGroup = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID_2)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .build();
    otherArtifact = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID_2)
        .setVersion(VERSION)
        .build();
    otherVersion = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION_2)
        .build();
  }

  @Test
  void isPlugin() {
    assertThat(descriptor.isPlugin(), is(false));
    assertThat(pluginDescriptor.isPlugin(), is(true));
  }

  @Test
  void getProperties() {
    assertThat(descriptor.getProperties().isEmpty(), is(true));
    assertThat(propertiedDescriptor.getProperties().isEmpty(), is(false));
    assertThrows(UnsupportedOperationException.class, () -> propertiedDescriptor.getProperties().put("bar", "blatch"));
  }

  @Test
  void testToString() {
    final String result = descriptor.toString();
    assertThat(result, startsWith("BundleDescriptor"));
    assertThat(result, containsString(GROUP_ID));
    assertThat(result, containsString(ARTIFACT_ID));
    assertThat(result, containsString(VERSION));
  }

  @ParameterizedTest(name = "{index}: equals for object")
  @MethodSource("equals_data")
  void testEquals(BundleDescriptor o1, Object o2, boolean expectedResult) {
    assertThat(o1.equals(o2), is(expectedResult));
  }

  @ParameterizedTest(name = "{index}: hashCode for object")
  @MethodSource("equals_data")
  void testHashCode(BundleDescriptor original, Object other, boolean expected) {
    assertThat(original.hashCode() == (other == null ? 0 : other.hashCode()), is(expected));
  }


  static List<Arguments> equals_data() {
    return asList(
                  arguments(descriptor, null, false),
                  arguments(descriptor, new Object(), false),
                  arguments(descriptor, descriptor, true),
                  arguments(descriptor, pluginDescriptor, false),
                  arguments(descriptor, propertiedDescriptor, true),
                  arguments(descriptor, otherGroup, false),
                  arguments(descriptor, otherArtifact, false),
                  arguments(descriptor, otherVersion, false));
  }
}
