/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;


import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Describes a bundle by its Maven coordinates.
 *
 * @since 2.0.0
 */
public class BundleDescriptor {

  private String groupId;
  private String artifactId;
  private String version;
  private String baseVersion;
  private String type = "jar";
  private Optional<String> classifier = empty();
  private Optional<String> systemPath = empty();
  private Optional<String> optional = empty();
  private Map<String, String> properties = emptyMap();
  private List<ArtifactCoordinates> exclusions = new ArrayList<>();

  protected BundleDescriptor() {}

  public String getGroupId() {
    return this.groupId;
  }

  public String getArtifactId() {
    return this.artifactId;
  }

  public String getVersion() {
    return this.version;
  }

  public String getBaseVersion() {
    return this.baseVersion;
  }

  public String getType() {
    return type;
  }

  public Optional<String> getClassifier() {
    return classifier;
  }

  public Optional<String> getSystemPath() {
    return systemPath;
  }

  public Optional<String> getOptional() {
    return optional;
  }

  public boolean isPlugin() {
    return classifier.map(classifierName -> classifierName.equals("mule-plugin")).orElse(false);
  }

  public Map<String, String> getProperties() {
    return unmodifiableMap(properties);
  }

  public List<ArtifactCoordinates> getExclusions() {
    return exclusions;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    BundleDescriptor that = (BundleDescriptor) o;

    if (!groupId.equals(that.groupId)) {
      return false;
    }
    if (!artifactId.equals(that.artifactId)) {
      return false;
    }
    if (!version.equals(that.version)) {
      return false;
    }

    return classifier.equals(that.classifier);
  }

  @Override
  public int hashCode() {
    int result = groupId.hashCode();
    result = 31 * result + artifactId.hashCode();
    result = 31 * result + version.hashCode();
    result = 31 * result + classifier.hashCode();
    return result;
  }

  @Override
  public String toString() {
    return "BundleDescriptor{" +
        "groupId='" + groupId + '\'' +
        ", artifactId='" + artifactId + '\'' +
        ", baseVersion='" + baseVersion + '\'' +
        ", version='" + version + '\'' +
        ", type='" + type + '\'' +
        ", classifier=" + classifier +
        ", systemPath=" + systemPath +
        '}';
  }

  /**
   * Builder for creating a {@code BundleDescriptor}
   */
  public static class Builder {

    private static final String ARTIFACT_ID = "artifact id";
    private static final String VERSION = "version";
    private static final String BASE_VERSION = "baseVersion";
    private static final String GROUP_ID = "group id";
    private static final String TYPE = "type";
    private static final String REQUIRED_FIELD_NOT_FOUND_TEMPLATE = "bundle cannot be created with null or empty %s";

    private BundleDescriptor bundleDescriptor = new BundleDescriptor();

    /**
     * @param groupId the group id of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setGroupId(String groupId) {
      validateIsNotEmpty(groupId, GROUP_ID);
      bundleDescriptor.groupId = groupId;
      return this;
    }

    /**
     * @param artifactId the artifactId id of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setArtifactId(String artifactId) {
      validateIsNotEmpty(artifactId, ARTIFACT_ID);
      bundleDescriptor.artifactId = artifactId;
      return this;
    }

    /**
     * This is the version of the bundle.
     *
     * @param version the version of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setVersion(String version) {
      validateIsNotEmpty(version, VERSION);
      bundleDescriptor.version = version;
      return this;
    }

    /**
     * Sets the base version of the bundle, for example "1.0-SNAPSHOT". In contrast to the {@link #getVersion()}, the base version
     * will always refer to the unresolved meta version.
     *
     * @param baseVersion the base version of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setBaseVersion(String baseVersion) {
      validateIsNotEmpty(baseVersion, BASE_VERSION);
      bundleDescriptor.baseVersion = baseVersion;
      return this;
    }

    /**
     * Sets the extension type of the bundle.
     *
     * @param type the type id of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setType(String type) {
      validateIsNotEmpty(type, TYPE);
      bundleDescriptor.type = type;
      return this;
    }

    /**
     * Sets the classifier of the bundle.
     *
     * @param classifier classifier of the bundle. Cannot be empty
     * @return the builder
     */
    public Builder setClassifier(String classifier) {
      bundleDescriptor.classifier = ofNullable(classifier);
      return this;
    }


    /**
     * Sets the system path of the bundle.
     *
     * @param systemPath systemPath of the bundle.
     * @return the builder
     */
    public Builder setSystemPath(String systemPath) {
      bundleDescriptor.systemPath = ofNullable(systemPath);
      return this;
    }

    /**
     * Sets the optional property of the bundle.
     *
     * @param optional optional property of the bundle.
     * @return the builder
     */
    public Builder setOptional(String optional) {
      bundleDescriptor.optional = ofNullable(optional);
      return this;
    }

    /**
     * Sets properties for the descriptor to be used by 3rd party.
     *
     * @param properties properties/metadata to associated to the bundle.
     * @return the builder
     */
    public Builder setProperties(Map<String, String> properties) {
      bundleDescriptor.properties = properties == null ? emptyMap() : new HashMap<>(properties);
      return this;
    }

    public Builder setExclusions(List<ArtifactCoordinates> exclusions) {
      bundleDescriptor.exclusions = exclusions;
      return this;
    }

    /**
     * @return a {@code BundleDescriptor} with the previous provided parameters to the builder.
     */
    public BundleDescriptor build() {
      validateIsNotEmpty(bundleDescriptor.groupId, GROUP_ID);
      validateIsNotEmpty(bundleDescriptor.artifactId, ARTIFACT_ID);
      validateIsNotEmpty(bundleDescriptor.version, VERSION);

      return this.bundleDescriptor;
    }

    private String getNullFieldMessage(String field) {
      return format(REQUIRED_FIELD_NOT_FOUND_TEMPLATE, field);
    }

    private void validateIsNotEmpty(String value, String fieldId) {
      if (isEmpty(value)) {
        throw new IllegalStateException(getNullFieldMessage(fieldId));
      }
    }

    private static boolean isEmpty(String value) {
      return value == null || value.equals("");
    }

  }
}
