/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

import java.net.URI;
import java.util.List;

import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.startsWith;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.params.provider.Arguments.arguments;

class BundleDependencyTestCase {

  private static final String GROUP_ID = "groupId1";
  private static final String ARTIFACT_ID = "artifactId1";
  private static final String GROUP_ID_2 = "groupId2";
  private static final String VERSION = "version1";
  private static BundleDependency dependency;
  private static BundleDependency differentScope;
  private static BundleDependency differentUri;
  private static BundleDependency differentDescriptor;
  private static BundleDependency usingBundleDescriptor;

  @BeforeAll
  static void setup() {
    final BundleDescriptor bundleDescriptor = new BundleDescriptor.Builder()
        .setGroupId(GROUP_ID)
        .setArtifactId(ARTIFACT_ID)
        .setVersion(VERSION)
        .setClassifier("mule-plugin")
        .build();
    dependency = new BundleDependency.Builder()
        .setDescriptor(bundleDescriptor)
        .setScope(BundleScope.TEST)
        .setBundleUri(URI.create("https://bandicoot.com"))
        .build();
    differentScope = new BundleDependency.Builder()
        .setDescriptor(bundleDescriptor)
        .setScope(BundleScope.COMPILE)
        .setBundleUri(URI.create("https://bandicoot.com"))
        .build();
    differentUri = new BundleDependency.Builder()
        .setDescriptor(bundleDescriptor)
        .setScope(BundleScope.TEST)
        .setBundleUri(URI.create("https://bilby.com"))
        .build();
    differentDescriptor = new BundleDependency.Builder()
        .setDescriptor(new BundleDescriptor.Builder()
            .setGroupId(GROUP_ID_2)
            .setArtifactId(ARTIFACT_ID)
            .setVersion(VERSION)
            .build())
        .setScope(BundleScope.TEST)
        .setBundleUri(URI.create("https://bandicoot.com"))
        .build();
    usingBundleDescriptor = new BundleDependency.Builder()
        .setBundleDescriptor(bundleDescriptor)
        .setScope(BundleScope.TEST)
        .setBundleUri(URI.create("https://bandicoot.com"))
        .build();
  }

  @Test
  void testToString() {
    final String result = dependency.toString();
    assertThat(result, startsWith("BundleDependency"));
    assertThat(result, containsString(GROUP_ID));
    assertThat(result, containsString(ARTIFACT_ID));
    assertThat(result, containsString(VERSION));
  }

  @ParameterizedTest(name = "{index}: equals for object")
  @MethodSource("equals_data")
  void testEquals(BundleDependency o1, Object o2, boolean expectedResult) {
    assertThat(o1.equals(o2), is(expectedResult));
  }

  @ParameterizedTest(name = "{index}: hashCode for object")
  @MethodSource("equals_data")
  void testHashCode(BundleDependency original, Object other, boolean expected) {
    assertThat(original.hashCode() == (other == null ? 0 : other.hashCode()), is(expected));
  }

  static List<Arguments> equals_data() {
    return asList(
                  arguments(dependency, null, false),
                  arguments(dependency, new Object(), false),
                  arguments(dependency, dependency, true),
                  arguments(dependency, usingBundleDescriptor, true),
                  arguments(dependency, differentUri, true),
                  arguments(dependency, differentScope, false),
                  arguments(dependency, differentDescriptor, false));
  }
}
