/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api;

import java.nio.file.Path;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;

/**
 * SPI for the API to find the implementation for the {@link MavenPomParser}
 *
 * @since 2.0
 */
public interface MavenPomParserProvider {

  /**
   * Creates a new {@link MavenPomParser} implementation.
   *
   * @pomPath pomFile {@link Path} pointing to the pom.
   * @return a new {@link MavenPomParser}
   */
  MavenPomParser createMavenPomParserClient(Path pomPath);

  /**
   * Creates a new {@link MavenPomParser} implementation.
   *
   * @param pomPath        {@link Path} pointing to the pom.
   * @param activeProfiles a {@link List} containing the active profiles
   * @return a new {@link MavenPomParser}
   */
  MavenPomParser createMavenPomParserClient(Path pomPath, List<String> activeProfiles);

  /**
   * Discovers a {@link MavenPomParserProvider}.
   *
   * @return a new client configured with the provided maven configuration.
   */
  static MavenPomParserProvider discoverProvider() {
    ServiceLoader<MavenPomParserProvider> factories = ServiceLoader.load(MavenPomParserProvider.class,
                                                                         MavenPomParserProvider.class.getClassLoader());
    Iterator<MavenPomParserProvider> iterator = factories.iterator();
    if (!iterator.hasNext()) {
      throw new IllegalStateException(String.format("Could not find %s service implementation through SPI",
                                                    MavenPomParserProvider.class.getName()));
    }
    return iterator.next();
  }

}
