/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import java.nio.file.Path;
import java.util.Properties;

/**
 * Build a {@link MavenPomModel} programmatically.
 *
 * @since 2.0.0
 */
public interface MavenModelBuilder {


  /**
   * Adds an additionalPluginDependencies to the pom model. If the plugin dependency does not exist yet in the mule-maven-plugin
   * configurations then it will create it.
   *
   * @param additionalPluginDependencies the additional dependencies to be added.
   */
  void addAdditionalPluginDependency(AdditionalPluginDependencies additionalPluginDependencies);

  /**
   * Adds a shared library to the pom model. If the plugin does not exist yet in the model then it will create it.
   *
   * @param groupId    the groupId of the dependency
   * @param artifactId the artifactId of the dependency
   *
   * @apiNote Only for compatibility. AdditionalPluginDependencies must be used instead.
   */
  void addSharedLibraryDependency(String groupId, String artifactId);

  /**
   * Creates the pom file for a deployable artifact inside the artifact exploded folder
   *
   * @param artifactFolder the deployable artifact folder.
   */
  void createDeployablePomFile(Path artifactFolder);

  /**
   * Creates the pom properties file for a deployable artifact inside the artifact exploded folder
   *
   * @param artifactFolder the deployable artifact folder
   * @param properties     the properties to write in the file
   */
  void createDeployablePomProperties(Path artifactFolder, Properties properties);

  /**
   * Updates the pom file from an artifact with the current model being built. If the file is a folder it will look up the pom
   * inside that folder. If it is a file it must be the pom.xml.
   *
   * @param path the path to the pom or to the folder containing the pom.
   */
  void updateArtifactPom(Path path);

  /**
   * Adds a new dependency to the application by locating the dependency and it's pom into the application repository and also
   * updating the pom file with the new dependency.
   *
   * @param bundleDependency dependency bundle descriptor.
   */
  void addDependency(BundleDependency bundleDependency);

  /**
   * Adds a repository to the {@link MavenPomModel}
   *
   * @param id   the repository id
   * @param name the repository name
   * @param url  the repository URL
   */
  void addRepository(String id, String name, String url);

  /**
   * Get the current {@link MavenPomModel}
   *
   * @return the {@link MavenPomModel}
   */
  MavenPomModel getModel();

  /**
   * Creates a new {@link MavenProfileBuilder}
   *
   * @return the new {@link MavenProfileBuilder}
   */
  MavenProfileBuilder getNewMavenProfileBuilder();

  /**
   * Builder to add a new Profile to the {@link MavenPomModel}
   */
  interface MavenProfileBuilder {

    /**
     * If set to true, this profile will be active unless another profile in this pom is activated.
     *
     * @param activeByDefault
     */
    MavenProfileBuilder setActiveByDefault(boolean activeByDefault);

    /**
     * Specifies for which JAVA version this profile will be activated.
     *
     * @param jdk the JAVA version.
     */
    MavenProfileBuilder setActivationByJdk(String jdk);

    /**
     * Adds a repository to the profile
     *
     * @param id   the repository id
     * @param name the repository name
     * @param url  the repository URL
     */
    MavenProfileBuilder addRepository(String id, String name, String url);

    /**
     * Adds a new dependency to the profile
     *
     * @param bundleDependency dependency bundle descriptor.
     */
    MavenProfileBuilder addDependency(BundleDependency bundleDependency);

    /**
     * Adds the profile to the {@link MavenPomModel}
     */
    void build();
  }
}
