/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.api.model;

import static java.lang.String.format;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Describes a dependency on a bundle.
 *
 * @since 2.0.0
 */
public class BundleDependency {

  private BundleDescriptor descriptor;
  private BundleScope scope;
  private URI bundleUri;
  private List<BundleDependency> transitiveDependencies = new ArrayList<>();

  protected BundleDependency() {}

  public BundleScope getScope() {
    return scope;
  }

  public BundleDescriptor getDescriptor() {
    return descriptor;
  }

  public URI getBundleUri() {
    return bundleUri;
  }

  public List<BundleDependency> getTransitiveDependencies() {
    return transitiveDependencies;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    BundleDependency that = (BundleDependency) o;
    return descriptor.equals(that.descriptor) && scope == that.scope;
  }

  @Override
  public int hashCode() {
    return Objects.hash(descriptor, scope != null ? scope.name() : null);
  }

  @Override
  public String toString() {
    return "BundleDependency{" +
        "descriptor=" + descriptor +
        ", scope=" + scope +
        ", bundleUrl=" + bundleUri +
        '}';
  }

  /**
   * Builder for creating a {@link BundleDependency}
   */
  public static class Builder {


    private static final String BUNDLE_DESCRIPTOR = "bundle descriptor";
    private static final String REQUIRED_FIELD_NOT_FOUND_TEMPLATE = "bundle cannot be created with null or empty %s";
    private static final String REQUIRED_FIELD_IS_NULL = "bundle cannot be created with null %s";

    private BundleDependency bundleDependency = new BundleDependency();

    /**
     * This is the descriptor of the bundle.
     *
     * @param descriptor the version of the bundle. Cannot be null or empty.
     * @return the builder
     */
    public Builder setDescriptor(BundleDescriptor descriptor) {
      validateIsNotNull(descriptor, BUNDLE_DESCRIPTOR);
      this.bundleDependency.descriptor = descriptor;

      return this;
    }

    /**
     * Sets the scope of the bundle.
     *
     * @param scope scope of the bundle. Non null
     * @return the builder
     */
    public Builder setScope(BundleScope scope) {
      checkState(scope != null, "scope cannot be null");
      bundleDependency.scope = scope;

      return this;
    }

    /**
     * Sets the descriptor of the bundle.
     *
     * @param descriptor describes the bundle which is dependent on. Cannot be null
     * @return the builder
     */
    public Builder setBundleDescriptor(BundleDescriptor descriptor) {
      setDescriptor(descriptor);

      return this;
    }

    public Builder setBundleUri(URI bundleUri) {
      validateIsNotNull(bundleUri, "Bundle URI cannot be null");
      this.bundleDependency.bundleUri = bundleUri;
      return this;
    }

    public Builder addTransitiveDependency(BundleDependency bundleDependency) {
      validateIsNotNull(bundleDependency, "Bundle dependency cannot be null");
      this.bundleDependency.transitiveDependencies.add(bundleDependency);
      return this;
    }

    /**
     * @return a {@code BundleDescriptor} with the previous provided parameters to the builder.
     */
    public BundleDependency build() {
      validateIsNotNull(bundleDependency.descriptor, BUNDLE_DESCRIPTOR);

      return this.bundleDependency;
    }

    private String getNullFieldMessage(String field) {
      return format(REQUIRED_FIELD_IS_NULL, field);
    }

    private void validateIsNotNull(Object value, String fieldId) {
      checkState(value != null, getNullFieldMessage(fieldId));
    }

    private static void checkState(boolean condition, String message) {
      if (!condition) {
        throw new IllegalStateException(message);
      }
    }
  }
}
