/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal;

import static org.mule.maven.pom.parser.api.model.BundleScope.valueOf;
import static org.mule.maven.pom.parser.internal.util.MavenUtils.getPomModel;

import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Stream.concat;

import static org.codehaus.plexus.util.StringUtils.isEmpty;
import static org.codehaus.plexus.util.xml.Xpp3DomUtils.mergeXpp3Dom;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.SharedLibrary;
import org.mule.maven.pom.parser.internal.model.MavenPomModelWrapper;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.stream.Stream;

import org.apache.maven.model.Build;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.FileSet;
import org.apache.maven.model.Model;
import org.apache.maven.model.Plugin;
import org.codehaus.plexus.util.xml.Xpp3Dom;

public class MavenPomParserImpl implements MavenPomParser {

  private static final String MULE_MAVEN_PLUGIN_GROUP_ID = "org.mule.tools.maven";
  private static final String MULE_MAVEN_PLUGIN_ARTIFACT_ID = "mule-maven-plugin";
  private static final String MULE_EXTENSIONS_PLUGIN_GROUP_ID = "org.mule.runtime.plugins";
  private static final String MULE_EXTENSIONS_PLUGIN_ARTIFACT_ID = "mule-extensions-maven-plugin";
  private static final String GROUP_ID = "groupId";
  private static final String ARTIFACT_ID = "artifactId";
  public static final String SHARED_LIBRARIES_FIELD = "sharedLibraries";
  public static final String SHARED_LIBRARY_FIELD = "sharedLibrary";
  public static final String DEFAULT_SOURCES_DIRECTORY = "src/main";
  private static final String VERSION = "version";
  private static final String MAVEN_SHADE_PLUGIN_ARTIFACT_ID = "maven-shade-plugin";
  private static final String ORG_APACHE_MAVEN_PLUGINS_GROUP_ID = "org.apache.maven.plugins";
  private static final String ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD = "additionalPluginDependencies";
  private static final String PLUGIN_FIELD = "plugin";
  private static final String PLUGIN_DEPENDENCY_FIELD = "dependency";
  private static final String PLUGIN_DEPENDENCIES_FIELD = "additionalDependencies";
  private static final String DEFAULT_ARTIFACT_TYPE = "jar";
  private final Model pomModel;
  private final List<String> activeProfiles;
  private final Build build;

  public MavenPomParserImpl(Path path) {
    this(path, emptyList());
  }

  public MavenPomParserImpl(Path path, List<String> activeProfiles) {
    requireNonNull(path, "path cannot be null");
    this.pomModel = getPomModel(path.toFile());
    this.activeProfiles = activeProfiles;
    this.build = pomModel.getBuild();
  }

  @Override
  public String getSourceDirectory() {
    return build.getSourceDirectory() != null ? build.getSourceDirectory() : DEFAULT_SOURCES_DIRECTORY;
  }

  @Override
  public List<String> getResourceDirectories() {
    return build.getResources().stream().map(FileSet::getDirectory).collect(toList());
  }

  @Override
  public MavenPomModel getModel() {
    return new MavenPomModelWrapper(pomModel);
  }

  @Override
  public List<SharedLibrary> getSharedLibraries() {
    List<SharedLibrary> shareLibrariesList = new LinkedList<>();
    findArtifactPackagerPlugin().ifPresent(plugin -> {
      Object configuration = plugin.getConfiguration();
      if (configuration != null) {
        Xpp3Dom sharedLibrariesDom = ((Xpp3Dom) configuration).getChild(SHARED_LIBRARIES_FIELD);
        if (sharedLibrariesDom != null) {
          Xpp3Dom[] sharedLibraries = sharedLibrariesDom.getChildren(SHARED_LIBRARY_FIELD);
          if (sharedLibraries != null) {
            for (Xpp3Dom sharedLibrary : sharedLibraries) {
              String groupId = getAttribute(sharedLibrary, GROUP_ID);
              String artifactId = getAttribute(sharedLibrary, ARTIFACT_ID);
              shareLibrariesList.add(new SharedLibrary(groupId, artifactId));
            }
          }
        }
      }
    });
    return unmodifiableList(shareLibrariesList);
  }

  @Override
  public List<BundleDependency> getDependencies() {
    return pomModel.getDependencies().stream().map(this::toBundleDependency).collect(toList());
  }

  @Override
  public Map<ArtifactCoordinates, AdditionalPluginDependencies> getPomAdditionalPluginDependenciesForArtifacts() {
    return findArtifactPackagerPlugin().map(this::getAdditionalPluginDependencies).orElse(emptyMap());
  }

  @Override
  public Properties getProperties() {
    return pomModel.getProperties();
  }

  @Override
  public boolean isMavenShadePluginConfigured() {
    if (pomModel.getBuild() != null) {
      for (Plugin plugin : pomModel.getBuild().getPlugins()) {
        if (plugin.getGroupId().equals(ORG_APACHE_MAVEN_PLUGINS_GROUP_ID)
            && plugin.getArtifactId().equals(MAVEN_SHADE_PLUGIN_ARTIFACT_ID)) {
          return true;
        }
      }
    }

    return false;
  }

  private Optional<Plugin> findArtifactPackagerPlugin() {
    Stream<Plugin> basePlugin = Stream.empty();
    Build build = pomModel.getBuild();
    if (build != null) {
      basePlugin = findArtifactPackagerPlugin(build.getPlugins()).map(Stream::of).orElse(Stream.empty());
    }

    // Sort them so the processing is consistent with how Maven calculates the plugin configuration for the effective pom.
    final List<String> sortedActiveProfiles = activeProfiles
        .stream()
        .sorted(String::compareTo)
        .collect(toList());

    final Stream<Plugin> packagerConfigsForActivePluginsStream = pomModel.getProfiles().stream()
        .filter(profile -> sortedActiveProfiles.contains(profile.getId()))
        .map(profile -> findArtifactPackagerPlugin(profile.getBuild() != null ? profile.getBuild().getPlugins() : null))
        .filter(plugin -> !plugin.equals(empty()))
        .map(Optional::get);

    return concat(basePlugin, packagerConfigsForActivePluginsStream)
        .reduce((p1, p2) -> {
          p1.setConfiguration(mergeXpp3Dom((Xpp3Dom) p2.getConfiguration(), (Xpp3Dom) p1.getConfiguration()));
          p1.getDependencies().addAll(p2.getDependencies());

          return p1;
        });
  }

  private Optional<Plugin> findArtifactPackagerPlugin(List<Plugin> plugins) {
    if (plugins != null) {
      return plugins.stream().filter(plugin -> (plugin.getArtifactId().equals(MULE_MAVEN_PLUGIN_ARTIFACT_ID)
          && plugin.getGroupId().equals(MULE_MAVEN_PLUGIN_GROUP_ID)) ||
          (plugin.getArtifactId().equals(MULE_EXTENSIONS_PLUGIN_ARTIFACT_ID) &&
              plugin.getGroupId().equals(MULE_EXTENSIONS_PLUGIN_GROUP_ID)))
          .findFirst();
    } else {
      return empty();
    }
  }


  private String getAttribute(Xpp3Dom tag, String attributeName) {
    Xpp3Dom attributeDom = tag.getChild(attributeName);
    checkState(attributeDom != null, format("'%s' element not declared at '%s' in the pom file",
                                            attributeName, tag.toString()));
    String attributeValue = attributeDom.getValue().trim();
    checkState(!isEmpty(attributeValue),
               format("'%s' was defined but has an empty value at '%s' declared in the pom file",
                      attributeName, tag.toString()));
    return attributeValue;
  }

  private Map<ArtifactCoordinates, AdditionalPluginDependencies> getAdditionalPluginDependencies(org.apache.maven.model.Plugin packagingPlugin) {
    Map<ArtifactCoordinates, AdditionalPluginDependencies> pluginsAdditionalLibraries = new HashMap<>();
    Object configuration = packagingPlugin.getConfiguration();
    if (configuration != null) {
      Xpp3Dom additionalPluginDependenciesDom = ((Xpp3Dom) configuration).getChild(ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD);
      if (additionalPluginDependenciesDom != null) {
        Xpp3Dom[] pluginsDom = additionalPluginDependenciesDom.getChildren(PLUGIN_FIELD);
        if (pluginsDom != null) {
          for (Xpp3Dom pluginDom : pluginsDom) {
            String pluginGroupId = getChildParameterValue(pluginDom, GROUP_ID, true);
            String pluginArtifactId = getChildParameterValue(pluginDom, ARTIFACT_ID, true);
            List<BundleDescriptor> additionalDependencyDependencies = new ArrayList<>();
            Xpp3Dom dependenciesDom = pluginDom.getChild(PLUGIN_DEPENDENCIES_FIELD);
            if (dependenciesDom != null) {
              for (Xpp3Dom dependencyDom : dependenciesDom.getChildren(PLUGIN_DEPENDENCY_FIELD)) {
                BundleDescriptor.Builder dependency = new BundleDescriptor.Builder();
                dependency.setGroupId(getChildParameterValue(dependencyDom, GROUP_ID, true));
                dependency
                    .setArtifactId(getChildParameterValue(dependencyDom, ARTIFACT_ID, true));
                dependency.setVersion(getChildParameterValue(dependencyDom, VERSION, true));
                String type = getChildParameterValue(dependencyDom, "type", false);
                dependency.setType(type == null ? DEFAULT_ARTIFACT_TYPE : type);
                dependency.setClassifier(getChildParameterValue(dependencyDom, "classifier", false));
                dependency.setSystemPath(getChildParameterValue(dependencyDom, "systemPath", false));

                additionalDependencyDependencies.add(dependency.build());
              }
            }
            AdditionalPluginDependencies additionalPluginDependencies =
                new AdditionalPluginDependencies(pluginGroupId, pluginArtifactId, additionalDependencyDependencies);
            pluginsAdditionalLibraries.put(new ArtifactCoordinates(pluginGroupId, pluginArtifactId),
                                           additionalPluginDependencies);
          }
        }
      }
    }
    return pluginsAdditionalLibraries;
  }

  private String getChildParameterValue(Xpp3Dom element, String childName, boolean validate) {
    Xpp3Dom child = element.getChild(childName);
    String childValue = child != null ? child.getValue() : null;
    if (isEmpty(childValue) && validate) {
      throw new IllegalArgumentException("Expecting child element with not null value " + childName);
    }
    return childValue;
  }

  private BundleDependency toBundleDependency(Dependency dependency) {
    BundleDescriptor descriptor = new BundleDescriptor.Builder()
        .setGroupId(dependency.getGroupId())
        .setArtifactId(dependency.getArtifactId())
        .setVersion(dependency.getVersion())
        .setType(dependency.getType())
        .setExclusions(dependency.getExclusions().stream().map(exclusion -> new ArtifactCoordinates(exclusion.getGroupId(),
                                                                                                    exclusion.getArtifactId()))
            .collect(toList()))
        .setClassifier(dependency.getClassifier())
        .setOptional(dependency.getOptional())
        .setSystemPath(dependency.getSystemPath())
        .build();
    BundleDependency.Builder builder = new BundleDependency.Builder().setDescriptor(descriptor);
    if (dependency.getScope() != null && !dependency.getScope().isEmpty()) {
      builder.setScope(valueOf(dependency.getScope().toUpperCase()));
    }
    return builder.build();
  }

  private void checkState(boolean condition, String message) {
    if (!condition) {
      throw new IllegalStateException(message);
    }
  }
}
