/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.model;

import static org.mule.maven.pom.parser.internal.MavenPomParserImpl.SHARED_LIBRARIES_FIELD;
import static org.mule.maven.pom.parser.internal.util.FileUtils.newFile;
import static org.mule.maven.pom.parser.internal.util.MavenUtils.getAttribute;
import static org.mule.maven.pom.parser.internal.util.MavenUtils.getPomModel;

import static java.io.File.separator;
import static java.lang.String.format;
import static java.nio.file.Paths.get;
import static java.util.Arrays.stream;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.filefilter.DirectoryFileFilter.DIRECTORY;

import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.MavenModelBuilder;
import org.mule.maven.pom.parser.api.model.MavenPomModel;

import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import org.apache.maven.model.Activation;
import org.apache.maven.model.Build;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Model;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.Profile;
import org.apache.maven.model.Repository;
import org.apache.maven.model.io.xpp3.MavenXpp3Writer;
import org.codehaus.plexus.util.xml.Xpp3Dom;

public class MavenModelBuilderImpl implements MavenModelBuilder {

  public static final String MULE_POM = "pom.xml";
  public static final String MULE_POM_PROPERTIES = "pom.properties";
  public static final String SHARED_LIBRARY_FIELD = "sharedLibrary";
  public static final String ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD = "additionalPluginDependencies";
  public static final String ADDITIONAL_DEPENDENCIES_FIELD = "additionalDependencies";
  public static final String DEPENDENCY_FIELD = "dependency";
  public static final String PLUGIN_FIELD = "plugin";
  public static final String VERSION_FIELD = "version";

  private static final String ARTIFACT_ID = "artifactId";
  private static final String GROUP_ID = "groupId";
  private static final String MULE_MAVEN_PLUGIN_ARTIFACT_ID = "mule-maven-plugin";
  private static final String MULE_MAVEN_PLUGIN_GROUP_ID = "org.mule.tools.maven";

  private final List<Dependency> dependencies = new ArrayList<>();
  private final Model model;

  public static MavenModelBuilder getDefaultModelBuilder() {
    return new MavenModelBuilderImpl("groupId", "artifactId", "1.0", empty(), empty());
  }

  public MavenModelBuilderImpl(String groupId, String artifactId, String version,
                               Optional<String> modelVersion, Optional<String> packaging) {
    requireNonNull(groupId, "groupId cannot be null");
    requireNonNull(artifactId, "artifactId cannot be null");
    requireNonNull(version, "version cannot be null");
    this.model = new Model();
    model.setGroupId(groupId);
    model.setArtifactId(artifactId);
    model.setVersion(version);
    model.setDependencies(dependencies);
    packaging.ifPresent(model::setPackaging);
    modelVersion.ifPresent(model::setModelVersion);
  }

  public MavenModelBuilderImpl(Path path) {
    requireNonNull(path, "path cannot be null");
    this.model = getPomModel(path.toFile());
  }

  @Override
  public void addAdditionalPluginDependency(AdditionalPluginDependencies additionalPluginDependencies) {
    Plugin mavenPlugin = getOrCreateMuleMavenPlugin();
    Xpp3Dom configuration = (Xpp3Dom) mavenPlugin.getConfiguration();
    if (configuration == null) {
      configuration = new Xpp3Dom("configuration");
      mavenPlugin.setConfiguration(configuration);
    }
    Xpp3Dom additionalPluginDependenciesDom = configuration.getChild(ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD);
    if (additionalPluginDependenciesDom == null) {
      additionalPluginDependenciesDom = new Xpp3Dom(ADDITIONAL_PLUGIN_DEPENDENCIES_FIELD);
      configuration.addChild(additionalPluginDependenciesDom);
    }

    Xpp3Dom[] pluginDoms = additionalPluginDependenciesDom.getChildren(PLUGIN_FIELD);
    Xpp3Dom finalAdditionalPluginDependenciesDom = additionalPluginDependenciesDom;
    Xpp3Dom additionalDependenciesForPlugin = stream(pluginDoms).filter(plugin -> {
      String libraryGroupId = getAttribute(plugin, "groupId");
      String libraryArtifactId = getAttribute(plugin, "artifactId");
      return additionalPluginDependencies.getGroupId().equals(libraryGroupId) &&
          additionalPluginDependencies.getArtifactId().equals(libraryArtifactId);
    }).findAny().orElseGet(() -> createPluginDom(finalAdditionalPluginDependenciesDom, additionalPluginDependencies));

    Xpp3Dom additionalDependenciesDom = getOrCreateAdditionalDependenciesDom(additionalDependenciesForPlugin);
    addAdditionalPluginDependency(additionalDependenciesDom, additionalPluginDependencies.getAdditionalDependencies());
  }

  @Override
  public void addSharedLibraryDependency(String groupId, String artifactId) {
    requireNonNull(groupId, "groupId cannot be null");
    requireNonNull(artifactId, "artifactId cannot be null");
    Plugin plugin = getOrCreateMuleMavenPlugin();
    Xpp3Dom configuration = (Xpp3Dom) plugin.getConfiguration();
    if (configuration == null) {
      configuration = new Xpp3Dom("configuration");
      plugin.setConfiguration(configuration);
    }

    Xpp3Dom sharedLibrariesDom = configuration.getChild(SHARED_LIBRARIES_FIELD);
    if (sharedLibrariesDom == null) {
      sharedLibrariesDom = new Xpp3Dom(SHARED_LIBRARIES_FIELD);
      configuration.addChild(sharedLibrariesDom);
    }

    Xpp3Dom[] sharedLibraries = sharedLibrariesDom.getChildren(SHARED_LIBRARY_FIELD);
    if (sharedLibraries != null) {
      Optional<Xpp3Dom> sharedLibrary = stream(sharedLibraries).filter(sharedLib -> {
        String libraryGroupId = getAttribute(sharedLib, "groupId");
        String libraryArtifactId = getAttribute(sharedLib, "artifactId");
        return groupId.equals(libraryGroupId) && artifactId.equals(libraryArtifactId);
      }).findAny();
      if (sharedLibrary.isPresent()) {
        return;
      }
    }

    Xpp3Dom sharedLibraryDom = new Xpp3Dom("sharedLibrary");
    sharedLibrariesDom.addChild(sharedLibraryDom);
    Xpp3Dom groupIdDom = new Xpp3Dom("groupId");
    groupIdDom.setValue(groupId);
    sharedLibraryDom.addChild(groupIdDom);
    Xpp3Dom artifactIdDom = new Xpp3Dom("artifactId");
    artifactIdDom.setValue(artifactId);
    sharedLibraryDom.addChild(artifactIdDom);
  }

  @Override
  public void createDeployablePomFile(Path artifactFolder) {
    requireNonNull(artifactFolder, "the artifactFolder's path cannot be null");
    File pomFileLocation =
        new File(artifactFolder.toFile(),
                 get("META-INF", "maven", model.getGroupId(), model.getArtifactId(), "pom.xml").toString());
    try {
      createFile(pomFileLocation.getAbsolutePath());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
    try (FileWriter fileWriter = new FileWriter(pomFileLocation)) {
      new MavenXpp3Writer().write(fileWriter, model);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public void createDeployablePomProperties(Path artifactFolder, Properties properties) {
    requireNonNull(artifactFolder, "the artifactFolder's path cannot be null");
    requireNonNull(properties, "property cannot be null");
    File pomPropertiesFileLocation =
        new File(artifactFolder.toFile(),
                 get("META-INF", "maven", model.getGroupId(), model.getArtifactId(), MULE_POM_PROPERTIES).toString());
    if (!pomPropertiesFileLocation.exists()) {
      try {
        createFile(pomPropertiesFileLocation.getAbsolutePath());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
    try (FileWriter fileWriter = new FileWriter(pomPropertiesFileLocation)) {
      properties.store(fileWriter, "Writing pom.properties");
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public void updateArtifactPom(Path path) {
    requireNonNull(path, "path cannot be null");
    File mulePluginPom = path.toFile();
    if (mulePluginPom.isDirectory()) {
      mulePluginPom = lookupPomFromMavenLocation(mulePluginPom);
    }
    MavenXpp3Writer writer = new MavenXpp3Writer();
    try (FileOutputStream outputStream = new FileOutputStream(mulePluginPom)) {
      writer.write(outputStream, model);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public void addDependency(BundleDependency bundleDependency) {
    requireNonNull(bundleDependency, "The bundleDependency cannot be null");
    model.addDependency(createMavenDependency(bundleDependency));
  }

  @Override
  public void addRepository(String id, String name, String url) {
    Repository repository = new Repository();
    repository.setId(id);
    repository.setName(name);
    repository.setUrl(url);
    model.addRepository(repository);
  }

  @Override
  public MavenPomModel getModel() {
    return new MavenPomModelWrapper(model);
  }

  @Override
  public MavenProfileBuilder getNewMavenProfileBuilder() {
    return new MavenProfileBuilderImpl();
  }

  private Plugin getOrCreateMuleMavenPlugin() {
    Build build = model.getBuild();
    if (build == null) {
      build = new Build();
      model.setBuild(build);
    }
    List<Plugin> plugins = build.getPlugins();
    if (plugins == null) {
      plugins = new ArrayList<>();
      build.setPlugins(plugins);
    }
    Optional<Plugin> pluginOptional = plugins.stream().filter(plugin -> plugin.getGroupId().equals(MULE_MAVEN_PLUGIN_GROUP_ID)
        && plugin.getArtifactId().equals(MULE_MAVEN_PLUGIN_ARTIFACT_ID)).findFirst();
    List<Plugin> finalPlugins = plugins;
    Plugin plugin = pluginOptional.orElseGet(() -> {
      Plugin muleMavenPlugin = new Plugin();
      muleMavenPlugin.setGroupId(MULE_MAVEN_PLUGIN_GROUP_ID);
      muleMavenPlugin.setArtifactId(MULE_MAVEN_PLUGIN_ARTIFACT_ID);
      finalPlugins.add(muleMavenPlugin);
      return muleMavenPlugin;
    });
    return plugin;
  }

  private Xpp3Dom getOrCreateAdditionalDependenciesDom(Xpp3Dom pluginDom) {
    Xpp3Dom additionalDependenciesDom = pluginDom.getChild(ADDITIONAL_DEPENDENCIES_FIELD);
    if (additionalDependenciesDom == null) {
      additionalDependenciesDom = new Xpp3Dom(ADDITIONAL_DEPENDENCIES_FIELD);
      pluginDom.addChild(additionalDependenciesDom);
    }
    return additionalDependenciesDom;
  }

  private void addAdditionalPluginDependency(Xpp3Dom additionalDependenciesDom, List<BundleDescriptor> descriptors) {
    Xpp3Dom[] dependencyDoms = additionalDependenciesDom.getChildren(DEPENDENCY_FIELD);
    descriptors.forEach(descriptor -> {
      Optional<Xpp3Dom> dependency = stream(dependencyDoms).filter(d -> {
        String dependencyGroupId = getAttribute(d, "groupId");
        String dependencyArtifactId = getAttribute(d, "artifactId");
        return descriptor.getGroupId().equals(dependencyGroupId) &&
            descriptor.getArtifactId().equals(dependencyArtifactId);
      }).findAny();
      if (dependency.isPresent()) {

        if (!descriptor.getVersion().equals(getAttribute(dependency.get(), "version"))) {
          throw new RuntimeException("Trying to add an additionalDependency already added with a different version");
        }
        return;
      }
      Xpp3Dom dependencyDom = new Xpp3Dom(DEPENDENCY_FIELD);
      additionalDependenciesDom.addChild(dependencyDom);

      Xpp3Dom groupIdDom = new Xpp3Dom(GROUP_ID);
      groupIdDom.setValue(descriptor.getGroupId());
      dependencyDom.addChild(groupIdDom);

      Xpp3Dom artifactIdDom = new Xpp3Dom(ARTIFACT_ID);
      artifactIdDom.setValue(descriptor.getArtifactId());
      dependencyDom.addChild(artifactIdDom);

      Xpp3Dom versionDom = new Xpp3Dom(VERSION_FIELD);
      versionDom.setValue(descriptor.getVersion());
      dependencyDom.addChild(versionDom);
    });
  }

  private Xpp3Dom createPluginDom(Xpp3Dom additionalPluginDependenciesDom,
                                  AdditionalPluginDependencies additionalPluginDependencies) {
    Xpp3Dom pluginDom = new Xpp3Dom(PLUGIN_FIELD);
    additionalPluginDependenciesDom.addChild(pluginDom);
    Xpp3Dom groupIdDom = new Xpp3Dom("groupId");
    groupIdDom.setValue(additionalPluginDependencies.getGroupId());
    pluginDom.addChild(groupIdDom);
    Xpp3Dom artifactIdDom = new Xpp3Dom("artifactId");
    artifactIdDom.setValue(additionalPluginDependencies.getArtifactId());
    pluginDom.addChild(artifactIdDom);
    return pluginDom;
  }

  private File createFile(String filename) throws IOException {
    File file = newFile(filename);
    if (!file.canWrite()) {
      String dirName = file.getPath();
      int i = dirName.lastIndexOf(File.separator);
      if (i > -1) {
        dirName = dirName.substring(0, i);
        File dir = newFile(dirName);
        dir.mkdirs();
      }
      file.createNewFile();
    }
    return file;
  }

  private File lookupPomFromMavenLocation(File artifactFolder) {
    File artifactPomFile = null;
    File lookupFolder = new File(artifactFolder, "META-INF" + separator + "maven");
    while (lookupFolder != null && lookupFolder.exists()) {
      File possiblePomLocation = new File(lookupFolder, MULE_POM);
      if (possiblePomLocation.exists()) {
        artifactPomFile = possiblePomLocation;
        break;
      }
      File[] directories = lookupFolder.listFiles((FileFilter) DIRECTORY);
      checkState(directories != null || directories.length == 0,
                 format("No directories under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      checkState(directories.length == 1,
                 format("More than one directory under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      lookupFolder = directories[0];
    }

    // final File artifactPomFile = new File(artifactFolder, MULE_ARTIFACT_FOLDER + separator + MULE_POM);
    if (artifactPomFile == null || !artifactPomFile.exists()) {
      throw new RuntimeException(format("The Maven bundle loader requires the file pom.xml (error found while reading artifact '%s')",
                                        artifactFolder.getName()));
    }
    return artifactPomFile;
  }

  private org.apache.maven.model.Dependency createMavenDependency(BundleDependency bundleDependency) {
    BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
    org.apache.maven.model.Dependency mavenModelDependency = new org.apache.maven.model.Dependency();
    mavenModelDependency.setGroupId(bundleDescriptor.getGroupId());
    mavenModelDependency.setArtifactId(bundleDescriptor.getArtifactId());
    mavenModelDependency.setVersion(bundleDescriptor.getVersion());
    mavenModelDependency.setType(bundleDescriptor.getType());

    bundleDescriptor.getClassifier().ifPresent(mavenModelDependency::setClassifier);
    bundleDescriptor.getOptional().ifPresent(mavenModelDependency::setOptional);

    if (bundleDependency.getScope() != null) {
      mavenModelDependency.setScope(bundleDependency.getScope().name().toLowerCase());
    }

    bundleDescriptor.getSystemPath().ifPresent(mavenModelDependency::setSystemPath);

    mavenModelDependency.setExclusions(bundleDescriptor.getExclusions().stream().map(exclusion -> {
      org.apache.maven.model.Exclusion mavenModelExclusion = new org.apache.maven.model.Exclusion();
      mavenModelExclusion.setGroupId(exclusion.getGroupId());
      mavenModelExclusion.setArtifactId(exclusion.getArtifactId());
      return mavenModelExclusion;
    }).collect(toList()));
    return mavenModelDependency;
  }

  private void checkState(boolean condition, String message) {
    if (!condition) {
      throw new IllegalStateException(message);
    }
  }

  public class MavenProfileBuilderImpl implements MavenProfileBuilder {

    private final Profile profile;
    private final Activation activation;

    public MavenProfileBuilderImpl() {
      this.profile = new Profile();
      this.activation = new Activation();
      profile.setActivation(activation);
    }

    @Override
    public MavenProfileBuilder setActiveByDefault(boolean activeByDefault) {
      activation.setActiveByDefault(activeByDefault);
      return this;
    }

    @Override
    public MavenProfileBuilder setActivationByJdk(String jdk) {
      activation.setJdk(jdk);
      return this;
    }

    @Override
    public MavenProfileBuilder addRepository(String id, String name, String url) {
      Repository repository = new Repository();
      repository.setId(id);
      repository.setName(name);
      repository.setUrl(url);
      profile.addRepository(repository);
      return this;
    }

    @Override
    public MavenProfileBuilder addDependency(BundleDependency bundleDependency) {
      Dependency mavenDependency = createMavenDependency(bundleDependency);
      profile.addDependency(mavenDependency);
      return this;
    }

    @Override
    public void build() {
      model.addProfile(profile);
    }
  }
}
