/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.util;

import static org.mule.maven.pom.parser.internal.util.FileUtils.getPomUrlFromJar;
import static org.mule.maven.pom.parser.internal.util.FileUtils.loadFileContentFrom;

import static java.io.File.separator;
import static java.lang.String.format;

import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.codehaus.plexus.util.StringUtils.isEmpty;

import org.mule.maven.pom.parser.api.BundleDescriptorCreationException;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;

import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

public class MavenUtils {

  private static final String META_INF = "META-INF";
  private static final String MULE_ARTIFACT_FOLDER = META_INF + separator + "mule-artifact";
  private static final String MULE_PLUGIN_POM = "pom.xml";

  /**
   * @param artifact a maven artifact that may be a directory following the mule deployable archive structure a compressed file or
   *                 file pointing to a pom.
   * @return the pom model for the artifact.
   */
  public static Model getPomModel(File artifact) {
    if (artifact.isDirectory()) {
      return getPomModelFolder(artifact);
    }
    if (getExtension(artifact.getName()).equals("jar")) {
      URL pomURL = getPomUrlFromJar(artifact);
      return getPomModelFromJar(pomURL);
    }
    return getPomModelFromFile(artifact);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param artifactFolder folder containing the exploded artifact file.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the folder does not contain a pom file or the file can' be loaded
   */
  public static Model getPomModelFolder(File artifactFolder) {
    final File mulePluginPom = lookupPomFromMavenLocation(artifactFolder);
    return getPomModelFromFile(mulePluginPom);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param pomUrl url pointing to the artifact content.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the artifact jar does not contain a pom file or the file can' be loaded
   */
  public static Model getPomModelFromJar(URL pomUrl) {
    String pomFilePath = MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM;
    try {
      MavenXpp3Reader reader = new MavenXpp3Reader();
      Model model = reader.read(new ByteArrayInputStream(loadFileContentFrom(pomUrl).get()));
      return model;
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' for the artifact '%s'",
                                                         pomFilePath, pomUrl.getFile()),
                                                  e);
    }
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param pomFile file containing the pom content.
   * @return the {@link Model} from the pom file
   * @throws BundleDescriptorCreationException if the file does not exist or the file can't be loaded
   */
  public static Model getPomModelFromFile(File pomFile) {
    MavenXpp3Reader reader = new MavenXpp3Reader();
    Model model;
    try (FileReader mulePluginPomFilerReader = new FileReader(pomFile)) {
      model = reader.read(mulePluginPomFilerReader);
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' in '%s'",
                                                         pomFile.getName(), pomFile.getParentFile().getAbsolutePath()),
                                                  e);
    }
    model.setPomFile(pomFile);
    return model;
  }

  private static File lookupPomFromMavenLocation(File artifactFolder) {
    File mulePluginPom = null;
    File lookupFolder = new File(artifactFolder, "META-INF" + File.separator + "maven");
    while (lookupFolder != null && lookupFolder.exists()) {
      File possiblePomLocation = new File(lookupFolder, MULE_PLUGIN_POM);
      if (possiblePomLocation.exists()) {
        mulePluginPom = possiblePomLocation;
        break;
      }
      File[] directories = lookupFolder.listFiles((FileFilter) DirectoryFileFilter.DIRECTORY);
      checkState(directories != null && directories.length > 0,
                 format("No directories under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      checkState(directories.length == 1,
                 format("More than one directory under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.getAbsolutePath(), artifactFolder.getAbsolutePath()));
      lookupFolder = directories[0];
    }

    // final File mulePluginPom = new File(artifactFolder, MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM);
    if (mulePluginPom == null || !mulePluginPom.exists()) {
      throw new BundleDescriptorCreationException(format("The maven bundle loader requires the file pom.xml (error found while reading artifact '%s')",
                                                         artifactFolder.getName()));
    }
    return mulePluginPom;
  }

  public static String getAttribute(Xpp3Dom tag, String attributeName) {
    Xpp3Dom attributeDom = tag.getChild(attributeName);
    checkState(attributeDom != null, format("'%s' element not declared at '%s' in the pom file",
                                            attributeName, tag.toString()));
    String attributeValue = attributeDom.getValue().trim();
    checkState(!isEmpty(attributeValue),
               format("'%s' was defined but has an empty value at '%s' declared in the pom file",
                      attributeName, tag.toString()));
    return attributeValue;
  }

  private static void checkState(boolean condition, String message) {
    if (!condition) {
      throw new IllegalStateException(message);
    }
  }
}
