/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal;

import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.MAVEN_PARSER_CLIENT;
import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.PomParsingStory.MAVEN_POM_PARSING_STORY;
import static org.mule.maven.pom.parser.internal.util.TestUtils.getPath;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.aMapWithSize;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasEntry;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.hasSize;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.AdditionalPluginDependencies;
import org.mule.maven.pom.parser.api.model.ArtifactCoordinates;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;
import org.mule.maven.pom.parser.api.model.SharedLibrary;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(MAVEN_PARSER_CLIENT)
@Story(MAVEN_POM_PARSING_STORY)
public class MavenPomParserImplTestCase {

  private final static String BASIC_WITH_RESOURCES_POM = "basic-with-resources/pom.xml";
  private final static String BASIC_WITH_DIFFERENT_SOURCE_DIRECTORY_POM = "basic-with-different-source-directory/pom.xml";
  private final static String BASIC_POM = "basic/pom.xml";
  private final static String BASIC_DEP_WITH_EMPTY_SCOPES_POM = "basic-dep-with-empty-scope/pom.xml";
  private final static String SHARE_LIB_POM = "shared-lib/pom.xml";
  private final static String ADDITIONAL_DEPENDENCIES_POM = "additional-plugin-dependency/pom.xml";
  private final static String ADDITIONAL_DEPENDENCIES_SYSTEM_PROPERTY_POM =
      "additional-plugin-dependency-system-property/pom.xml";
  private final static String ADDITIONAL_DEPENDENCIES_PROPERTY_POM =
      "additional-plugin-dependency-property/pom.xml";
  private final static String BASIC_WITH_SHADE_PLUGIN_POM = "basic-with-shade-plugin/pom.xml";
  private final static String RECURSIVE_POM_RESOLUTION = "recursive-pom-resolution";
  private final static String RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES = RECURSIVE_POM_RESOLUTION + "/shared-libraries";
  private final static String RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES =
      RECURSIVE_POM_RESOLUTION + "/additional-plugin-dependencies";
  private final static String RECURSIVE_POM_RESOLUTION_APP = "%s/%s/level-1/level-2/with-parent-pom/pom.xml";

  @Test
  public void getSourceDirectoryWhenNotPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    assertThat(mavenPomParser.getSourceDirectory(), is(MavenPomParserImpl.DEFAULT_SOURCES_DIRECTORY));
  }

  @Test
  @Issue("W-15677833")
  public void getCompileDependenciesWithEmptyScope() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_DEP_WITH_EMPTY_SCOPES_POM));
    final List<BundleDependency> noScopeDeps = mavenPomParser.getDependencies().stream()
        .filter(dep -> dep.getScope() == null)
        .collect(toList());

    assertThat(noScopeDeps, hasSize(2));
  }

  @Test
  public void getSourceDirectoryWhenPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_WITH_DIFFERENT_SOURCE_DIRECTORY_POM));
    assertThat(mavenPomParser.getSourceDirectory(), is("src/main2"));
  }

  @Test
  public void getResourceDirectoryWhenPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_WITH_RESOURCES_POM));
    List<String> resourceDirectories = mavenPomParser.getResourceDirectories();
    assertThat(resourceDirectories, hasSize(2));
    assertThat(resourceDirectories, containsInAnyOrder("src/main/resources", "src/main/resources2"));
  }

  @Test
  public void getResourceDirectoryWhenNotPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    assertThat(mavenPomParser.getResourceDirectories(), is(empty()));
  }

  @Test
  public void getSharedLibrariesTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(SHARE_LIB_POM));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("org.apache.derby"));
    assertThat(sharedLibrary.getArtifactId(), is("derby"));
  }

  @Test
  public void getDependencies() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    List<BundleDependency> dependencies = mavenPomParser.getDependencies();
    assertThat(dependencies, hasSize(3));
    List<BundleDescriptor> descriptors = dependencies.stream().map(BundleDependency::getDescriptor).collect(toList());
    assertThat(descriptors, hasItems(
                                     allOf(hasProperty("groupId", is("org.mule.connectors")),
                                           hasProperty("artifactId", is("mule-http-connector")),
                                           hasProperty("version", is("1.1.0"))),
                                     allOf(hasProperty("groupId", is("org.mule.connectors")),
                                           hasProperty("artifactId", is("mule-db-connector")),
                                           hasProperty("version", is("1.13.1"))),
                                     allOf(hasProperty("groupId", is("org.mule.connectors")),
                                           hasProperty("artifactId", is("mule-jms-connector")),
                                           hasProperty("version", is("1.8.4")))));
  }

  @Test
  public void getSharedLibrariesWhenNotPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(0));
  }

  @Test
  public void getPomAdditionalDependenciesTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(ADDITIONAL_DEPENDENCIES_POM));
    List<AdditionalPluginDependencies> artifactsAdditionalDependencies =
        new LinkedList<>(mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts().values());
    assertThat(artifactsAdditionalDependencies, hasSize(1));
    AdditionalPluginDependencies mavenPluginAdditionalDependencies = artifactsAdditionalDependencies.get(0);
    assertThat(mavenPluginAdditionalDependencies.getGroupId(), is("org.mule.connectors"));
    assertThat(mavenPluginAdditionalDependencies.getArtifactId(), is("mule-db-connector"));
    List<BundleDescriptor> additionalDependencies = mavenPluginAdditionalDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(1));
    BundleDescriptor bundleDescriptor = additionalDependencies.get(0);
    assertThat(bundleDescriptor.getGroupId(), is("org.apache.derby"));
    assertThat(bundleDescriptor.getArtifactId(), is("derby"));
    assertThat(bundleDescriptor.getVersion(), is("10.14.2.0"));
  }

  @Test
  public void getPomAdditionalDependenciesSystemPropertyTestCase() {
    System.setProperty("derbyVersion", "10.14.2.0");

    try {
      MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(ADDITIONAL_DEPENDENCIES_SYSTEM_PROPERTY_POM));
      List<AdditionalPluginDependencies> artifactsAdditionalDependencies =
          new LinkedList<>(mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts().values());
      assertThat(artifactsAdditionalDependencies, hasSize(1));
      AdditionalPluginDependencies mavenPluginAdditionalDependencies = artifactsAdditionalDependencies.get(0);
      assertThat(mavenPluginAdditionalDependencies.getGroupId(), is("org.mule.connectors"));
      assertThat(mavenPluginAdditionalDependencies.getArtifactId(), is("mule-db-connector"));
      List<BundleDescriptor> additionalDependencies = mavenPluginAdditionalDependencies.getAdditionalDependencies();
      assertThat(additionalDependencies, hasSize(1));
      BundleDescriptor bundleDescriptor = additionalDependencies.get(0);
      assertThat(bundleDescriptor.getGroupId(), is("org.apache.derby"));
      assertThat(bundleDescriptor.getArtifactId(), is("derby"));
      assertThat(bundleDescriptor.getVersion(), is("10.14.2.0"));
    } finally {
      System.clearProperty("derbyVersion");
    }
  }

  @Test
  @Issue("W-16190426")
  public void getPomAdditionalDependenciesPropertyTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(ADDITIONAL_DEPENDENCIES_PROPERTY_POM));
    List<AdditionalPluginDependencies> artifactsAdditionalDependencies =
        new LinkedList<>(mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts().values());
    assertThat(artifactsAdditionalDependencies, hasSize(1));
    AdditionalPluginDependencies mavenPluginAdditionalDependencies = artifactsAdditionalDependencies.get(0);
    assertThat(mavenPluginAdditionalDependencies.getGroupId(), is("org.mule.connectors"));
    assertThat(mavenPluginAdditionalDependencies.getArtifactId(), is("mule-db-connector"));
    List<BundleDescriptor> additionalDependencies = mavenPluginAdditionalDependencies.getAdditionalDependencies();
    assertThat(additionalDependencies, hasSize(1));
    BundleDescriptor bundleDescriptor = additionalDependencies.get(0);
    assertThat(bundleDescriptor.getGroupId(), is("org.apache.derby"));
    assertThat(bundleDescriptor.getArtifactId(), is("derby"));
    assertThat(bundleDescriptor.getVersion(), is("10.14.2.0"));
  }

  @Test
  public void getPomAdditionalDependenciesPropertyPrecedenceTestCase() {
    System.setProperty("derbyVersion", "10.15.1.3");

    try {
      MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(ADDITIONAL_DEPENDENCIES_PROPERTY_POM));
      List<AdditionalPluginDependencies> artifactsAdditionalDependencies =
          new LinkedList<>(mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts().values());
      assertThat(artifactsAdditionalDependencies, hasSize(1));
      AdditionalPluginDependencies mavenPluginAdditionalDependencies = artifactsAdditionalDependencies.get(0);
      assertThat(mavenPluginAdditionalDependencies.getGroupId(), is("org.mule.connectors"));
      assertThat(mavenPluginAdditionalDependencies.getArtifactId(), is("mule-db-connector"));
      List<BundleDescriptor> additionalDependencies = mavenPluginAdditionalDependencies.getAdditionalDependencies();
      assertThat(additionalDependencies, hasSize(1));
      BundleDescriptor bundleDescriptor = additionalDependencies.get(0);
      assertThat(bundleDescriptor.getGroupId(), is("org.apache.derby"));
      assertThat(bundleDescriptor.getArtifactId(), is("derby"));
      // system property overrides the one defined in the pom
      assertThat(bundleDescriptor.getVersion(), is("10.15.1.3"));
    } finally {
      System.clearProperty("derbyVersion");
    }
  }

  @Test
  public void getPomAdditionalDependenciesWhenNotPresentTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    Collection<AdditionalPluginDependencies> artifactsAdditionalDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts().values();
    assertThat(artifactsAdditionalDependencies, hasSize(0));
  }

  @Test
  public void isShadePluginConfiguredTestCase() {
    MavenPomParser mavenPomParser = new MavenPomParserImpl(getPath(BASIC_POM));
    assertThat(mavenPomParser.isMavenShadePluginConfigured(), is(false));

    mavenPomParser = new MavenPomParserImpl(getPath(BASIC_WITH_SHADE_PLUGIN_POM));
    assertThat(mavenPomParser.isMavenShadePluginConfigured(), is(true));
  }

  @Test
  @Issue("W-19885318")
  public void getDependenciesFromAppAndPrentAndGrandparent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION + "/dependencies/",
                                              "in-app-and-parent-and-grandparent")));
    List<BundleDependency> dependencies = mavenPomParser.getDependencies();
    assertThat(dependencies, hasSize(3));
    List<BundleDescriptor> descriptors = dependencies.stream().map(BundleDependency::getDescriptor).collect(toList());
    assertThat(descriptors, contains(
                                     allOf(hasProperty("groupId", is("org.mule.connectors")),
                                           hasProperty("artifactId", is("mule-http-connector")),
                                           hasProperty("version", is("1.5.25"))),
                                     allOf(hasProperty("groupId", is("org.mule.tests.extensions")),
                                           hasProperty("artifactId", is("mule-petstore-extension")),
                                           hasProperty("version", is("4.10.0"))),
                                     allOf(hasProperty("groupId", is("org.mule.connectors")),
                                           hasProperty("artifactId", is("mule-db-connector")),
                                           hasProperty("version", is("1.13.1")))));
  }

  @Test
  @Issue("W-19885318")
  public void getPropertiesFromAppAndPrentAndGrandparent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION + "/properties/",
                                              "in-app-and-parent-and-grandparent")));
    Properties properties = mavenPomParser.getProperties();
    assertThat(properties, aMapWithSize(5));
    assertThat(properties, allOf(
                                 hasEntry("grandparent1", "grandparent1"),
                                 hasEntry("grandparent2", "grandparent2"),
                                 hasEntry("parent", "parent"),
                                 hasEntry("app", "app"),
                                 hasEntry("repeated", "app")));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromAppWhenInAppAndPrentAndGrandparent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES,
                                              "/in-app-and-parent-and-grandparent-takes-app/")));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("app"));
    assertThat(sharedLibrary.getArtifactId(), is("app"));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromParentWhenInParentAndGrandParent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES,
                                              "/in-parent-and-grandparent-takes-parent/")));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("parent"));
    assertThat(sharedLibrary.getArtifactId(), is("parent"));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromParentWhenInParentOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES,
                                              "/in-parent-only-takes-parent/")));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("parent"));
    assertThat(sharedLibrary.getArtifactId(), is("parent"));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromGrandparentWhenInGrandparentOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP, RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES,
                                              "/in-grandparent-only-takes-grandparent/")));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("grandparent"));
    assertThat(sharedLibrary.getArtifactId(), is("grandparent"));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromPluginsWhenInPluginsAndPluginsManagement() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES
            + "/in-plugins-and-plugins-management-takes-plugins-only/pom.xml"));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("plugins"));
    assertThat(sharedLibrary.getArtifactId(), is("plugins"));
  }

  @Test
  @Issue("W-19885318")
  public void getSharedLibrariesFromPluginsManagementWhenInPluginsManagementOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(RECURSIVE_POM_RESOLUTION_SHARED_LIBRARIES
            + "/in-plugins-management-only-takes-plugins-management/pom.xml"));
    List<SharedLibrary> sharedLibraries = mavenPomParser.getSharedLibraries();
    assertThat(sharedLibraries, hasSize(1));
    SharedLibrary sharedLibrary = sharedLibraries.get(0);
    assertThat(sharedLibrary.getGroupId(), is("plugins-management"));
    assertThat(sharedLibrary.getArtifactId(), is("plugins-management"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromAppWhenInAppAndPrentAndGrandparent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP,
                                              RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES,
                                              "/in-app-and-parent-and-grandparent-takes-app/")));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("app"));
    assertThat(entry.getKey().getArtifactId(), is("app"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromParentWhenInParentAndGrandParent() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP,
                                              RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES,
                                              "/in-parent-and-grandparent-takes-parent/")));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("parent"));
    assertThat(entry.getKey().getArtifactId(), is("parent"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromParentWhenInParentOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP,
                                              RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES,
                                              "/in-parent-only-takes-parent/")));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("parent"));
    assertThat(entry.getKey().getArtifactId(), is("parent"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromGrandparentWhenInGrandparentOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(format(RECURSIVE_POM_RESOLUTION_APP,
                                              RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES,
                                              "/in-grandparent-only-takes-grandparent/")));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("grandparent"));
    assertThat(entry.getKey().getArtifactId(), is("grandparent"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromPluginsWhenInPluginsAndPluginsManagement() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES +
            "/in-plugins-and-plugins-management-takes-plugins-only/pom.xml"));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("plugins"));
    assertThat(entry.getKey().getArtifactId(), is("plugins"));
  }

  @Test
  @Issue("W-19885318")
  public void getAdditionalPluginDependenciesFromPluginsManagementWhenInPluginsManagementOnly() {
    MavenPomParser mavenPomParser =
        new MavenPomParserImpl(getPath(RECURSIVE_POM_RESOLUTION_ADDITIONAL_PLUGIN_DEPENDENCIES +
            "/in-plugins-management-only-takes-plugins-management/pom.xml"));
    Map<ArtifactCoordinates, AdditionalPluginDependencies> additionalPluginDependencies =
        mavenPomParser.getPomAdditionalPluginDependenciesForArtifacts();
    assertThat(additionalPluginDependencies, aMapWithSize(1));
    Map.Entry<ArtifactCoordinates, AdditionalPluginDependencies> entry =
        additionalPluginDependencies.entrySet().iterator().next();
    assertThat(entry.getKey().getGroupId(), is("plugins-management"));
    assertThat(entry.getKey().getArtifactId(), is("plugins-management"));
  }
}
