/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal.util;

import static org.mule.maven.pom.parser.internal.util.FileUtils.getPomUrlFromJar;
import static org.mule.maven.pom.parser.internal.util.FileUtils.loadFileContentFrom;

import static java.io.File.separator;
import static java.lang.String.format;
import static java.nio.file.Files.exists;
import static java.nio.file.Files.isDirectory;
import static java.nio.file.Files.list;
import static java.nio.file.Files.newBufferedReader;
import static java.util.stream.Collectors.toList;

import static org.apache.commons.io.FilenameUtils.getExtension;
import static org.codehaus.plexus.util.StringUtils.isEmpty;

import org.mule.maven.pom.parser.api.BundleDescriptorCreationException;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.Reader;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Stream;

import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;

public class MavenUtils {

  private static final String META_INF = "META-INF";
  private static final String MULE_ARTIFACT_FOLDER = META_INF + separator + "mule-artifact";
  private static final String MULE_PLUGIN_POM = "pom.xml";

  /**
   * @param artifact a maven artifact that may be a directory following the mule deployable archive structure a compressed file or
   *                 file pointing to a pom.
   * @return the pom model for the artifact.
   */
  public static Model getPomModel(Path artifact) {
    if (isDirectory(artifact)) {
      return getPomModelFolder(artifact);
    }
    if (getExtension(artifact.getFileName().toString()).equals("jar")) {
      URL pomURL = getPomUrlFromJar(artifact);
      return getPomModelFromJar(pomURL);
    }
    return getPomModelFromFile(artifact);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param artifactFolder folder containing the exploded artifact file.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the folder does not contain a pom file or the file can' be loaded
   */
  public static Model getPomModelFolder(Path artifactFolder) {
    final Path mulePluginPom = lookupPomFromMavenLocation(artifactFolder);
    return getPomModelFromFile(mulePluginPom);
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param pomUrl url pointing to the artifact content.
   * @return the {@link Model} from the pom file if available
   * @throws BundleDescriptorCreationException if the artifact jar does not contain a pom file or the file can' be loaded
   */
  public static Model getPomModelFromJar(URL pomUrl) {
    String pomFilePath = MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM;
    try {
      MavenXpp3Reader reader = new MavenXpp3Reader();
      Model model = reader.read(new ByteArrayInputStream(loadFileContentFrom(pomUrl).get()));
      return model;
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' for the artifact '%s'",
                                                         pomFilePath, pomUrl.getFile()),
                                                  e);
    }
  }

  /**
   * Returns the {@link Model} from a given artifact folder
   *
   * @param pomFile file containing the pom content.
   * @return the {@link Model} from the pom file
   * @throws BundleDescriptorCreationException if the file does not exist or the file can't be loaded
   */
  public static Model getPomModelFromFile(Path pomFile) {
    MavenXpp3Reader reader = new MavenXpp3Reader();
    Model model;
    try (Reader mulePluginPomFilerReader = newBufferedReader(pomFile)) {
      model = reader.read(mulePluginPomFilerReader);
    } catch (IOException | XmlPullParserException e) {
      throw new BundleDescriptorCreationException(format("There was an issue reading '%s' in '%s'",
                                                         pomFile.getFileName().toString(),
                                                         pomFile.getParent().toAbsolutePath().toString()),
                                                  e);
    }
    model.setPomFile(pomFile.toFile());
    return model;
  }

  public static Path lookupPomFromMavenLocation(Path artifactFolder) {
    Path mulePluginPom = null;
    Path lookupFolder = artifactFolder.resolve("META-INF").resolve("maven");
    while (lookupFolder != null && exists(lookupFolder)) {
      Path possiblePomLocation = lookupFolder.resolve(MULE_PLUGIN_POM);
      if (exists(possiblePomLocation)) {
        mulePluginPom = possiblePomLocation;
        break;
      }

      List<Path> directories;
      try (Stream<Path> directoriesStream = list(lookupFolder)) {
        directories = directoriesStream
            .filter(Files::isDirectory)
            .collect(toList());
      } catch (IOException e) {
        throw new IllegalStateException(e);
      }

      checkState(directories != null && !directories.isEmpty(),
                 format("No directories under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.toAbsolutePath().toString(), artifactFolder.toAbsolutePath().toString()));
      checkState(directories.size() == 1,
                 format("More than one directory under %s so pom.xml file for artifact in folder %s could not be found",
                        lookupFolder.toAbsolutePath().toString(), artifactFolder.toAbsolutePath().toString()));
      lookupFolder = directories.get(0);
    }

    // final File mulePluginPom = new File(artifactFolder, MULE_ARTIFACT_FOLDER + separator + MULE_PLUGIN_POM);
    if (mulePluginPom == null || !exists(mulePluginPom)) {
      throw new IllegalStateException(format("The maven bundle loader requires the file pom.xml (error found while reading artifact '%s')",
                                             artifactFolder.getFileName().toString()));
    }
    return mulePluginPom;
  }

  public static String getAttribute(Xpp3Dom tag, String attributeName) {
    Xpp3Dom attributeDom = tag.getChild(attributeName);
    checkState(attributeDom != null, format("'%s' element not declared at '%s' in the pom file",
                                            attributeName, tag.toString()));
    String attributeValue = attributeDom.getValue().trim();
    checkState(!isEmpty(attributeValue),
               format("'%s' was defined but has an empty value at '%s' declared in the pom file",
                      attributeName, tag.toString()));
    return attributeValue;
  }

  private static void checkState(boolean condition, String message) {
    if (!condition) {
      throw new IllegalStateException(message);
    }
  }
}
