/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.pom.parser.internal;

import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.MAVEN_PARSER_CLIENT;
import static org.mule.maven.pom.parser.internal.AllureConstants.MavenParserClient.ModelBuilderStory.MAVEN_MODEL_BUILDER_STORY;
import static org.mule.maven.pom.parser.internal.util.TestUtils.getFile;

import static java.util.Optional.empty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

import org.mule.maven.pom.parser.api.model.MavenPomModel;
import org.mule.maven.pom.parser.api.model.PomParentCoordinates;

import java.nio.file.Path;
import java.nio.file.Paths;

import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(MAVEN_PARSER_CLIENT)
@Story(MAVEN_MODEL_BUILDER_STORY)
public class MavenPomModelTestCase {

  private final static Path BASIC_POM = Paths.get("basic", "pom.xml");
  private final static Path BASIC_WITH_PARENT_POM = Paths.get("basic-with-parent", "pom.xml");

  @Test
  public void getModelTestCase() {
    MavenPomModel model = new MavenPomParserImpl(getFile(BASIC_POM)).getModel();
    assertThat(model.getGroupId(), is("com.test"));
    assertThat(model.getArtifactId(), is("basic"));
    assertThat(model.getVersion(), is("1.0.0-SNAPSHOT"));
    assertThat(model.getPackaging(), is("mule-application"));
    assertThat(model.getPomFile().get().getPath(), containsString(BASIC_POM.toString()));
    assertThat(model.getParent(), is(empty()));
  }

  @Test
  public void getModelWithParentTestCase() {
    MavenPomModel model = new MavenPomParserImpl(getFile(BASIC_WITH_PARENT_POM)).getModel();
    assertThat(model.getGroupId(), is("com.test"));
    assertThat(model.getArtifactId(), is("basic-with-parent"));
    assertThat(model.getVersion(), is("1.0.0-SNAPSHOT"));
    assertThat(model.getPackaging(), is("mule-application"));
    assertThat(model.getPomFile().get().getPath(), containsString(BASIC_WITH_PARENT_POM.toString()));
    PomParentCoordinates parent = model.getParent().get();
    assertThat(parent.getGroupId(), is("com.test"));
    assertThat(parent.getArtifactId(), is("basic-parent"));
    assertThat(parent.getVersion(), is("1.0.0-SNAPSHOT"));
  }
}
