/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.annotation;


import org.mule.metadata.api.model.MetadataType;

import java.util.Map;

/**
 * Interface to be implemented to extend the functionality of {@link MetadataType} serialization. The implementation of
 * this class is highly recommended when using custom {@link TypeAnnotation}s in a {@link MetadataType} description to be
 * able to create more human readable serialization output. This implementations will be located using SPI.
 * <p>
 * The purpose of this registry is to provide a mapping between a "friendly name" ({@link TypeAnnotation#getName()}) and
 * the correspondent concrete class ({@link TypeAnnotation#getClass()}), this allows to the serialization to write a more
 * compact and easy to read name, instead of using the full qualifier name of the concrete class.
 * <p>
 * <b>Using this mapping the serialization goes from:</b>
 * <pre>{"org.mule.metadata.api.annotation.TypeIdAnnotation":"java.lang.String"}</pre>
 * <b>To:</b>
 * <pre>{"typeId":"java.lang.String"}</pre>
 * By default the {@link MetadataType} serialization will provide mappings for out of the box {@link TypeAnnotation}
 * implementations as {@link TypeIdAnnotation}, {@link LabelAnnotation}, etc.
 * <b>To contribute to the {@link MetadataType}'s {@link TypeAnnotation} serialization, follow this steps steps:</b>
 * <ol>
 * <li>Create an implementation of {@link AnnotationsRegistry} which must return a {@link Map} of {@link String} and
 * {@link Class} that extends from {@link TypeAnnotation}</li>
 * <li>Create a file with {@code org.mule.metadata.api.annotation.AnnotationsRegistry} located a {@code META-INF/services/}</li>
 * <li>Complete fill the file with the full qualifier name of your {@link AnnotationsRegistry} implementation</li>
 * </ol>
 * <b>Considerations:</b>
 * <ol>
 * <li>The name used as Key of the map for a class, should be the same in {@link TypeAnnotation#getName()}</li>
 * <li>If a mapping is not found for a given {@link TypeAnnotation} the full qualifier name will be used</li>
 * <li>If a mapping tries to use a taken name from an already provided {@link TypeAnnotation} implementation, the
 * serialization will fail</li>
 * <li>If the class defined for a {@link TypeAnnotation} implementation is not found in the in the ClassLoader, the
 * deserialization of this object will be ignored.</li>
 * <li>If many {@link AnnotationsRegistry} are found, all mappings will be merged</li>
 * </ol>
 */
public interface AnnotationsRegistry
{

    Map<String, Class<? extends TypeAnnotation>> getRegistry();
}
