/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.annotation.UniquesItemsAnnotation;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.utils.LazyValue;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

public class ArrayTypeBuilder<P extends TypeBuilder> extends AbstractBuilder<ArrayType> implements TypeBuilder<ArrayType>, WithAnnotation<ArrayTypeBuilder>
{

    private Optional<TypeBuilder<?>> arrayType;
    private Optional<MetadataType> typeValue;
    private LazyValue<ArrayType> result = new LazyValue<>();

    protected ArrayTypeBuilder(MetadataFormat format)
    {
        super(format);
        this.arrayType = Optional.empty();
        this.typeValue = Optional.empty();
    }

    public BaseTypeBuilder<ArrayTypeBuilder<P>> of()
    {
        final BaseTypeBuilder<ArrayTypeBuilder<P>> typeBuilder = new BaseTypeBuilder<>(format);
        this.arrayType = Optional.of(typeBuilder);
        return typeBuilder;
    }

    public ArrayTypeBuilder<P> id(String typeIdentifier)
    {
        return with(new TypeIdAnnotation(typeIdentifier));
    }

    public ArrayTypeBuilder<P> of(TypeBuilder builder)
    {
        this.arrayType = Optional.ofNullable(builder);
        return this;
    }

    public ArrayTypeBuilder<P> of(MetadataType typeValue)
    {
        this.typeValue = Optional.ofNullable(typeValue);
        return this;
    }

    public ArrayTypeBuilder<P> with(TypeAnnotation extension)
    {
        this.addExtension(extension);
        return this;
    }

    public ArrayTypeBuilder<P> boundary(Optional<? extends Number> minLength,
                                        Optional<? extends Number> maxLength)
    {
        return with(new LengthAnnotation(minLength, maxLength));
    }

    public ArrayTypeBuilder<P> description(String lang, String content)
    {
        return with(new DescriptionAnnotation(Optional.of(lang), content));
    }

    public ArrayTypeBuilder<P> description(String content)
    {
        return with(new DescriptionAnnotation(Optional.empty(), content));
    }

    public ArrayTypeBuilder<P> uniques()
    {
        return with(new UniquesItemsAnnotation());
    }

    public ArrayTypeBuilder<P> label(String label)
    {
        return with(new LabelAnnotation(label));
    }

    @Override
    public ArrayType build()
    {
        return typeValue.isPresent() ?
               new DefaultArrayType(() -> typeValue.get(), format, annotations) :
               buildArrayType(arrayType.orElseThrow(() -> new RuntimeException("Array type was not specified.")));
    }

    private ArrayType buildArrayType(TypeBuilder<?> arrayBuilder)
    {
        ArrayType type;
        AtomicReference<MetadataType> metadataTypeHolder = new AtomicReference<>();
        if (isRecursiveType(arrayBuilder) || result.isDefined())
        {
            type = result.get(() -> new DefaultArrayType(metadataTypeHolder::get, format, annotations));
            metadataTypeHolder.set(type);
        }
        else
        {
            metadataTypeHolder.set(arrayBuilder.build());
            type = result.get(() -> new DefaultArrayType(metadataTypeHolder::get, format, annotations));
        }
        return type;
    }

    private boolean isRecursiveType(TypeBuilder<?> typeBuilder)
    {
        return typeBuilder.equals(this);
    }
}
