/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import static java.util.stream.Collectors.toList;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.OpenObjectAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.utils.LazyValue;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class ObjectTypeBuilder<P extends TypeBuilder> extends AbstractBuilder<ObjectType> implements TypeBuilder<ObjectType>, WithAnnotation<ObjectTypeBuilder>
{

    private List<ObjectFieldTypeBuilder<?>> fieldsBuilder;
    private boolean ordered;
    private LazyValue<ObjectType> result;

    public ObjectTypeBuilder(MetadataFormat format)
    {
        super(format);
        this.fieldsBuilder = new ArrayList<>();
        this.result = new LazyValue<>();
    }

    public ObjectFieldTypeBuilder<ObjectTypeBuilder<P>> addField()
    {
        final ObjectFieldTypeBuilder<ObjectTypeBuilder<P>> fieldMetadataBuilder = new ObjectFieldTypeBuilder<>(format);
        this.fieldsBuilder.add(fieldMetadataBuilder);
        return fieldMetadataBuilder;
    }

    public ObjectTypeBuilder<P> ordered(boolean ordered)
    {
        this.ordered = ordered;
        return this;
    }

    public ObjectTypeBuilder<P> id(String typeIdentifier)
    {
        return with(new TypeIdAnnotation(typeIdentifier));
    }

    public ObjectTypeBuilder<P> with(TypeAnnotation extension)
    {
        this.addExtension(extension);
        return this;
    }

    public ObjectTypeBuilder<P> description(String lang, String content)
    {
        return with(new DescriptionAnnotation(Optional.of(lang), content));
    }

    public ObjectTypeBuilder<P> description(String content)
    {
        return with(new DescriptionAnnotation(Optional.empty(), content));
    }

    public ObjectTypeBuilder<P> label(String label)
    {
        return with(new LabelAnnotation(label));
    }

    public ObjectTypeBuilder<P> open()
    {
        return with(new OpenObjectAnnotation());
    }

    @Override
    public ObjectType build()
    {
        final ArrayList<ObjectFieldType> fields = new ArrayList<>();
        // This is to avoid stack overflow we create the object with no fields
        boolean needsInit = !result.isDefined();
        final ObjectType objectType = result.get(() -> new DefaultObjectType(fields, ordered, format, annotations));
        if (needsInit)
        {
            final List<ObjectFieldType> collect = fieldsBuilder.stream()
                    .map((fieldBuilder) -> fieldBuilder.build())
                    .collect(toList());
            fields.addAll(collect);
        }
        return objectType;
    }

}
