/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.utils;

import static org.mule.metadata.utils.MetadataTypeUtils.getSingleAnnotation;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.Optional;

final class TypeIdResolver
{

    private TypeIdResolver()
    {
    }

    static Optional<String> resolveTypeId(MetadataType metadataType)
    {
        TypeResolverVisitor visitor = new TypeResolverVisitor();
        metadataType.accept(visitor);

        return Optional.ofNullable(visitor.getResolvedTypeId());
    }

    private static class TypeResolverVisitor extends MetadataTypeVisitor
    {

        private String typeId = null;

        private TypeResolverVisitor()
        {
        }

        @Override
        public void visitString(StringType stringType)
        {
            Optional<EnumAnnotation> enumAnnotation = getSingleAnnotation(stringType, EnumAnnotation.class);
            if (enumAnnotation.isPresent())
            {
                defaultVisit(stringType);
            }
            else
            {
                typeId = String.class.getName();
            }
        }

        @Override
        public void visitObjectField(ObjectFieldType objectFieldType)
        {
            defaultVisit(objectFieldType.getValue());
        }

        @Override
        public void visitAnyType(AnyType anyType)
        {
            typeId = Object.class.getName();
        }

        @Override
        protected void defaultVisit(MetadataType metadataType)
        {
            typeId = getSingleAnnotation(metadataType, TypeIdAnnotation.class).map(TypeIdAnnotation::getValue).orElse(null);
        }

        public String getResolvedTypeId()
        {
            return typeId;
        }
    }
}
