/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.impl.DefaultAttributeFieldType;
import org.mule.metadata.api.model.impl.DefaultAttributeKeyType;

import java.util.Collections;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

public class AttributeFieldTypeBuilder<P extends TypeBuilder> extends AbstractBuilder<AttributeFieldType>
    implements TypeBuilder<AttributeFieldType> {

  private Optional<QName> name;
  private Optional<Pattern> pattern;
  private Optional<TypeBuilder<?>> value;
  private boolean required;

  protected AttributeFieldTypeBuilder(MetadataFormat format) {
    super(format);
    this.name = Optional.empty();
    this.pattern = Optional.empty();
    this.value = Optional.empty();
    this.required = false;
  }

  public AttributeFieldTypeBuilder<P> name(QName name) {
    this.name = Optional.of(name);
    return this;
  }

  public AttributeFieldTypeBuilder<P> name(String name) {
    this.name = Optional.of(new QName(name));
    return this;
  }

  public AttributeFieldTypeBuilder<P> pattern(Pattern pattern) {
    this.pattern = Optional.of(pattern);
    return this;
  }

  public BaseTypeBuilder<AttributeFieldTypeBuilder<P>> value() {
    final BaseTypeBuilder<AttributeFieldTypeBuilder<P>> typeBuilder = new BaseTypeBuilder<>(format);
    this.value = Optional.of(typeBuilder);
    return typeBuilder;
  }

  public AttributeFieldTypeBuilder<P> value(TypeBuilder baseTypeBuilder) {
    this.value = Optional.of(baseTypeBuilder);
    return this;
  }

  public void required(boolean required) {
    this.required = required;
  }

  @Override
  public AttributeFieldType build() {
    if (!name.isPresent() && !pattern.isPresent()) {
      throw new RuntimeException("No name or pattern was specified.");
    }
    final TypeBuilder<?> typeBuilder = this.value.orElseThrow(() -> new RuntimeException("No value was specified."));
    final MetadataType build = typeBuilder.build();
    if (!(build instanceof SimpleType)) {
      throw new RuntimeException("Value should be a simple type.");
    }
    final DefaultAttributeKeyType key = new DefaultAttributeKeyType(name, pattern, format, Collections.emptyMap());
    return new DefaultAttributeFieldType(key, (SimpleType) build, required, format, annotations);
  }

}
