/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.DictionaryType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultDictionaryType;

import java.util.Optional;

/**
 * Implementation of {@link TypeBuilder} that creates instances of {@link DictionaryType}
 *
 * @param <P>
 * @since 1.0
 */
public class DictionaryTypeBuilder<P extends TypeBuilder> extends AbstractBuilder<DictionaryType>
    implements TypeBuilder<DictionaryType>, WithAnnotation<DictionaryTypeBuilder> {

  private Optional<TypeBuilder<?>> keyTypeBuilder;
  private Optional<TypeBuilder<?>> valueTypeBuider;
  private Optional<MetadataType> keyType;
  private Optional<MetadataType> valueType;

  protected DictionaryTypeBuilder(MetadataFormat format) {
    super(format);
    keyTypeBuilder = Optional.empty();
    valueTypeBuider = Optional.empty();
    keyType = Optional.empty();
    valueType = Optional.empty();
  }

  public DictionaryTypeBuilder<P> id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public BaseTypeBuilder<?> ofKey() {
    final BaseTypeBuilder<?> typeBuilder = new BaseTypeBuilder<>(format);
    ofKey(typeBuilder);

    return typeBuilder;
  }

  public DictionaryTypeBuilder<P> ofKey(TypeBuilder builder) {
    keyTypeBuilder = Optional.ofNullable(builder);
    return this;
  }

  public DictionaryTypeBuilder<P> ofKey(MetadataType type) {
    keyType = Optional.ofNullable(type);
    return this;
  }

  public BaseTypeBuilder<?> ofValue() {
    final BaseTypeBuilder<?> typeBuilder = new BaseTypeBuilder<>(format);
    ofValue(typeBuilder);

    return typeBuilder;
  }

  public DictionaryTypeBuilder<P> ofValue(TypeBuilder builder) {
    valueTypeBuider = Optional.ofNullable(builder);
    return this;
  }

  public DictionaryTypeBuilder<P> ofValue(MetadataType type) {
    valueType = Optional.ofNullable(type);
    return this;
  }


  public DictionaryTypeBuilder<P> with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  public DictionaryTypeBuilder<P> boundary(Optional<? extends Number> minLength,
                                           Optional<? extends Number> maxLength) {
    return with(new LengthAnnotation(minLength, maxLength));
  }

  public DictionaryTypeBuilder<P> description(String lang, String content) {
    return with(new DescriptionAnnotation(Optional.of(lang), content));
  }

  public DictionaryTypeBuilder<P> description(String content) {
    return with(new DescriptionAnnotation(Optional.empty(), content));
  }

  public DictionaryTypeBuilder<P> label(String label) {
    return with(new LabelAnnotation(label));
  }

  @Override
  public DictionaryType build() {
    final MetadataType key = keyType.isPresent() ? keyType.get()
        : buildOrThrow(this.keyTypeBuilder, "Key type was not specified.");

    final MetadataType value = valueType.isPresent() ? valueType.get()
        : buildOrThrow(this.valueTypeBuider, "Value type was not specified.");

    return new DefaultDictionaryType(format, annotations, key, value);
  }

  private MetadataType buildOrThrow(Optional<TypeBuilder<?>> builder, String errorMessage) {
    return builder.orElseThrow(() -> new RuntimeException(errorMessage)).build();
  }
}
