/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.Accessibility;
import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.FieldOccurrenceAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.impl.DefaultObjectFieldType;
import org.mule.metadata.utils.LazyValue;

import java.util.Optional;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

public class ObjectFieldTypeBuilder<P extends TypeBuilder> extends AbstractBuilder<ObjectFieldType>
    implements TypeBuilder<ObjectFieldType>, WithAnnotation<ObjectFieldTypeBuilder> {

  private Optional<ObjectKeyBuilder<ObjectFieldTypeBuilder<P>>> keyBuilder;
  private Optional<? extends TypeBuilder<?>> valueBuilder;
  private boolean required;
  private boolean repeated;

  private LazyValue<ObjectFieldType> result;
  private MetadataType typeValue;

  protected ObjectFieldTypeBuilder(MetadataFormat format) {
    super(format);
    this.keyBuilder = Optional.empty();
    this.valueBuilder = Optional.empty();
    this.required = false;
    this.repeated = false;
    this.result = new LazyValue<>();
  }

  public ObjectFieldTypeBuilder<P> with(TypeAnnotation extension) {
    addExtension(extension);
    return this;
  }

  public ObjectFieldTypeBuilder<P> withKeyAnnotation(TypeAnnotation extension) {
    keyBuilder.ifPresent(builder -> builder.addExtension(extension));
    return this;
  }

  public ObjectFieldTypeBuilder<P> description(String lang, String content) {
    return with(new DescriptionAnnotation(Optional.of(lang), content));
  }

  public ObjectFieldTypeBuilder<P> description(String content) {
    return with(new DescriptionAnnotation(Optional.empty(), content));
  }

  public ObjectFieldTypeBuilder<P> required(boolean required) {
    this.required = required;
    return this;
  }

  public ObjectFieldTypeBuilder<P> repeated(boolean repeated) {
    this.repeated = repeated;
    return this;
  }

  public ObjectFieldTypeBuilder<P> accessibility(Accessibility accessibility) {
    return with(new AccessibilityAnnotation(accessibility));
  }

  public ObjectFieldTypeBuilder<P> occurrence(Optional<Number> min,
                                              Optional<Number> max) {
    return with(new FieldOccurrenceAnnotation(min, max));
  }

  public ObjectFieldTypeBuilder<P> label(String label) {
    withKeyAnnotation(new LabelAnnotation(label));
    return this;
  }

  public ObjectKeyBuilder<ObjectFieldTypeBuilder<P>> key(QName keyName) {
    final ObjectKeyBuilder<ObjectFieldTypeBuilder<P>> keyBuilder = new ObjectKeyBuilder<>(
                                                                                          format);
    keyBuilder.name(keyName);
    this.keyBuilder = Optional.of(keyBuilder);
    return keyBuilder;
  }

  public ObjectFieldTypeBuilder<P> key(String keyName) {
    final ObjectKeyBuilder<ObjectFieldTypeBuilder<P>> keyBuilder = new ObjectKeyBuilder<>(format);
    keyBuilder.name(keyName);
    this.keyBuilder = Optional.of(keyBuilder);
    return this;
  }

  public ObjectFieldTypeBuilder<P> key(Pattern pattern) {
    final ObjectKeyBuilder<ObjectFieldTypeBuilder<P>> keyBuilder = new ObjectKeyBuilder<>(
                                                                                          format);
    keyBuilder.pattern(pattern);
    this.keyBuilder = Optional.of(keyBuilder);
    return this;
  }

  public BaseTypeBuilder<ObjectFieldTypeBuilder<P>> value() {
    final Optional<BaseTypeBuilder<ObjectFieldTypeBuilder<P>>> typeBuilder = Optional.of(new BaseTypeBuilder<>(format));
    valueBuilder = typeBuilder;
    return typeBuilder.get();
  }


  public ObjectFieldTypeBuilder<P> value(MetadataType typeValue) {
    this.typeValue = typeValue;
    return this;
  }

  public ObjectFieldTypeBuilder<P> value(TypeBuilder<?> builder) {
    valueBuilder = Optional.of(builder);
    return this;
  }

  @Override
  public ObjectFieldType build() {
    return result.get(() -> {
      final ObjectKeyType keyType =
          keyBuilder.orElseThrow(() -> new RuntimeException("No field key was specified for building the ObjectType.")).build();
      final MetadataType value = typeValue != null ? typeValue
          : valueBuilder.orElseThrow(() -> new RuntimeException("No field value was specified for building the ObjectType."))
              .build();
      return new DefaultObjectFieldType(keyType, value, required, repeated, format, annotations);
    });
  }

}
