/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import static java.util.stream.Collectors.toList;
import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.impl.DefaultObjectKeyType;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

public class ObjectKeyBuilder<P extends TypeBuilder> extends AbstractBuilder<ObjectKeyType>
    implements TypeBuilder<ObjectKeyType> {

  private Optional<QName> name;
  private Optional<Pattern> pattern;
  private List<AttributeFieldTypeBuilder<?>> attributes;

  protected ObjectKeyBuilder(MetadataFormat format) {
    super(format);
    this.name = Optional.empty();
    this.pattern = Optional.empty();
    this.attributes = new ArrayList<>();
  }

  public ObjectKeyBuilder<P> name(String name) {
    this.name = Optional.of(new QName(name));
    return this;
  }

  public ObjectKeyBuilder<P> name(QName name) {
    this.name = Optional.of(name);
    return this;
  }

  public ObjectKeyBuilder<P> pattern(Pattern name) {
    this.pattern = Optional.of(name);
    return this;
  }

  public AttributeFieldTypeBuilder<ObjectKeyBuilder<P>> addAttribute() {
    final AttributeFieldTypeBuilder<ObjectKeyBuilder<P>> attributeBuilder = new AttributeFieldTypeBuilder<>(format);
    attributes.add(attributeBuilder);
    return attributeBuilder;
  }

  @Override
  public ObjectKeyType build() {
    if (!name.isPresent() && !pattern.isPresent()) {
      throw new RuntimeException("No name or pattern was specified.");
    }

    final List<AttributeFieldType> attributes = this.attributes.stream()
        .map((builder) -> builder.build())
        .collect(toList());

    return new DefaultObjectKeyType(name, pattern, attributes, format, annotations);
  }
}
