/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.handler;

import static java.beans.Introspector.getBeanInfo;
import static java.util.Arrays.asList;
import org.mule.metadata.api.annotation.Accessibility;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.utils.ParsingContext;
import org.mule.metadata.java.api.utils.TypeResolver;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class DefaultObjectFieldHandler implements ObjectFieldHandler
{
    @Override
    public void handleFields(Class<?> clazz,
                             TypeHandlerManager typeHandlerManager, ParsingContext context,
                             ObjectTypeBuilder builder)
    {
        if (!Object.class.equals(clazz) && !Void.class.equals(clazz) && !void.class.equals(clazz))
        {
            try
            {
                final List<PropertyDescriptor> propertyDescriptors = loadProperties(clazz);
                for (PropertyDescriptor pd : propertyDescriptors)
                {
                    final Class<?> propertyClass = pd.getPropertyType();
                    if (propertyClass != null)
                    { // property class returns null
                      // if there are getters with
                      // no associated field.
                        final ObjectFieldTypeBuilder<?> fieldBuilder = builder.addField();
                        final String name = pd.getName();
                        fieldBuilder.key(name);
                        final Optional<Accessibility> visibility = getVisibility(pd);
                        if (visibility.isPresent())
                        {
                            fieldBuilder.accessibility(visibility.get());
                        }
                        final Type propertyType = getPropertyType(pd);
                        // Check recursive declarations if it was already
                        // defined
                        final Optional<TypeBuilder<?>> typeBuilder = context.getTypeBuilder(propertyType);
                        if (typeBuilder.isPresent())
                        {
                            fieldBuilder.value(typeBuilder.get());
                        }
                        else
                        {
                            typeHandlerManager.handle(propertyType, context, fieldBuilder.value());
                        }
                    }
                }
            }
            catch (Exception e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    private Optional<Accessibility> getVisibility(PropertyDescriptor pd)
    {
        if (pd.getReadMethod() != null && pd.getWriteMethod() == null)
        {
            return Optional.of(Accessibility.READ_ONLY);
        }
        else if (pd.getReadMethod() == null && pd.getWriteMethod() != null)
        {
            return Optional.of(Accessibility.WRITE_ONLY);
        }
        return Optional.empty();
    }

    private Type getPropertyType(PropertyDescriptor pd)
    {
        final Type propertyType;
        if (pd.getReadMethod() != null)
        {
            propertyType = pd.getReadMethod().getGenericReturnType();
        }
        else
        {
            propertyType = pd.getWriteMethod().getParameterTypes()[0];
        }
        return propertyType;
    }

    private List<PropertyDescriptor> loadProperties(Class<?> clazz) throws IntrospectionException
    {
        final List<PropertyDescriptor> propertyDescriptors;
        if (clazz.isInterface())
        {
            propertyDescriptors = new ArrayList<>();
            propertyDescriptors.addAll(asList(getBeanInfo(clazz).getPropertyDescriptors()));
            for (Class<?> interfaceType : TypeResolver.getSuperInterfaces(clazz))
            {
                propertyDescriptors.addAll(asList(getBeanInfo(interfaceType).getPropertyDescriptors()));
            }
        }
        else
        {
            propertyDescriptors = asList(getBeanInfo(clazz, Object.class).getPropertyDescriptors());
        }
        return propertyDescriptors;
    }
}
