/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.handler;

import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.utils.ParsingContext;
import org.mule.metadata.java.api.utils.TypeResolver;
import org.mule.metadata.java.internal.handler.ArrayClassHandler;
import org.mule.metadata.java.internal.handler.BinaryHandler;
import org.mule.metadata.java.internal.handler.BooleanHandler;
import org.mule.metadata.java.internal.handler.CollectionClassHandler;
import org.mule.metadata.java.internal.handler.DateHandler;
import org.mule.metadata.java.internal.handler.DateTimeHandler;
import org.mule.metadata.java.internal.handler.EnumHandler;
import org.mule.metadata.java.internal.handler.MapClassHandler;
import org.mule.metadata.java.internal.handler.NumberHandler;
import org.mule.metadata.java.internal.handler.ObjectHandler;
import org.mule.metadata.java.internal.handler.StringHandler;

import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class TypeHandlerManager
{

    private final List<ClassHandler> handlers;

    private TypeHandlerManager(ClassHandler... handlers)
    {
        this.handlers = Arrays.asList(handlers);
    }

    public void handle(Type type, ParsingContext context, BaseTypeBuilder<?> typeBuilder)
    {

        if (type instanceof Class<?>)
        {
            final TypeBuilder<?> builder = handleClass((Class<?>) type, Collections.emptyList(), context, typeBuilder);
            context.addTypeBuilder(type, builder);
        }
        else if (type instanceof ParameterizedType)
        {
            // for parameterized type, discover raw type and resolved variables
            final ParameterizedType paramType = (ParameterizedType) type;
            final Class<?> raw = TypeResolver.erase(type);
            context.addResolvedVariables(TypeResolver.resolveVariables(type));
            // some other parameterized type, resolve variables and proceed with
            // raw type
            final TypeBuilder<?> builder = handleClass(raw, Arrays.asList(paramType.getActualTypeArguments()), context, typeBuilder);
            context.addTypeBuilder(type, builder);
        }
        else if (type instanceof GenericArrayType)
        {
            final GenericArrayType arrayType = (GenericArrayType) type;
            final ArrayTypeBuilder<?> arrayTypeBuilder = typeBuilder.arrayType();
            handle(arrayType.getGenericComponentType(), context, arrayTypeBuilder.of());
        }
        else if (type instanceof TypeVariable<?>)
        {
            final Type actual = context.getResolvedVariables().get(type);
            if (actual == null || type.equals(actual))
            {
                final TypeVariable<?> typeVariable = (TypeVariable<?>) type;
                final Type bounds[] = typeVariable.getBounds();
                if (bounds.length > 0)
                {
                    handle(bounds[0], context, typeBuilder);
                }
                else
                {
                    typeBuilder.anyType();
                }
            }
            else
            {
                handle(actual, context, typeBuilder);
            }
        }
        else if (type instanceof WildcardType)
        {
            final WildcardType wildType = (WildcardType) type;
            final Type lowerBounds[] = wildType.getLowerBounds();
            final Type upperBounds[] = wildType.getUpperBounds();
            if (lowerBounds.length > 0)
            {
                handle(lowerBounds[0], context, typeBuilder);
            }
            else if (upperBounds.length > 0)
            {
                handle(upperBounds[0], context, typeBuilder);
            }
            else
            {
                typeBuilder.anyType();
            }
        }
        else
        {
            throw new IllegalArgumentException("Unsupported type " + type);
        }
    }

    private TypeBuilder<?> handleClass(Class<?> type, List<Type> genericTypes, ParsingContext context, BaseTypeBuilder<?> typeBuilder)
    {
        for (ClassHandler handler : handlers)
        {
            if (handler.handles(type))
            {
                return handler.handleClass(type, genericTypes, this, context, typeBuilder);
            }
        }
        return typeBuilder.anyType();
    }

    public static TypeHandlerManager createDefault()
    {
        return create(new ObjectHandler(new DefaultObjectFieldHandler()));
    }

    public static TypeHandlerManager create(ObjectHandler objectHandler)
    {
        return new TypeHandlerManager(new BinaryHandler(), new DateHandler(),
                                      new DateTimeHandler(), new EnumHandler(),
                                      new MapClassHandler(), new NumberHandler(),
                                      new MapClassHandler(), new NumberHandler(),
                                      new StringHandler(), new ArrayClassHandler(),
                                      new BooleanHandler(), new CollectionClassHandler(),
                                      objectHandler);
    }

}
