/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.internal.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.DictionaryTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.handler.ClassHandler;
import org.mule.metadata.java.api.handler.TypeHandlerManager;
import org.mule.metadata.java.api.utils.ParsingContext;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class MapClassHandler implements ClassHandler {

  @Override
  public boolean handles(Class<?> clazz) {
    return Map.class.isAssignableFrom(clazz);
  }

  @Override
  public TypeBuilder<?> handleClass(Class<?> clazz,
                                    List<Type> genericTypes, TypeHandlerManager typeHandlerManager,
                                    ParsingContext context, BaseTypeBuilder<?> typeBuilder) {
    final DictionaryTypeBuilder<? extends BaseTypeBuilder<?>> dictionaryType = typeBuilder.dictionaryType().id(clazz.getName());

    if (genericTypes.isEmpty()) {
      dictionaryType.ofKey().anyType();
      dictionaryType.ofValue().anyType();
    } else if (genericTypes.size() != 2) {
      throw new IllegalArgumentException(String.format("Exactly 2 generics were expected but %d were provided instead",
                                                       genericTypes.size()));
    } else {
      parseGenericType(genericTypes.get(0), dictionaryType::ofKey, dictionaryType::ofKey, context, typeHandlerManager);
      parseGenericType(genericTypes.get(1), dictionaryType::ofValue, dictionaryType::ofValue, context, typeHandlerManager);
    }

    return dictionaryType;
  }

  private void parseGenericType(Type genericType,
                                Consumer<TypeBuilder<?>> typeConsumer,
                                Supplier<BaseTypeBuilder<?>> fallbackType,
                                ParsingContext context,
                                TypeHandlerManager typeHandlerManager) {

    final Optional<TypeBuilder<?>> keyTypeBuilder = context.getTypeBuilder(genericType);
    if (keyTypeBuilder.isPresent()) {
      typeConsumer.accept(keyTypeBuilder.get());
    } else {
      typeHandlerManager.handle(genericType, context, fallbackType.get());
    }
  }
}
