/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.example;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;

public class PrimitiveHandler implements JsonElementHandler {

  @Override
  public boolean handles(JsonElement jsonElement) {
    return jsonElement.isJsonPrimitive();
  }

  @Override
  public TypeBuilder<?> handle(JsonElement jsonElement, BaseTypeBuilder<?> root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {
    JsonPrimitive primitive = (JsonPrimitive) jsonElement;

    if (primitive.isString()) {
      return root.stringType();
    } else if (primitive.isBoolean()) {
      return root.booleanType();
    } else if (primitive.isNumber()) {
      //Check if it is an integer. If not return Number type
      String stringValue = primitive.getAsString();
      if (isInteger(stringValue)) {
        return root.numberType().id("int");
      }
      if (isDouble(stringValue)) {
        return root.numberType().id("double");
      } else {
        return root.numberType();
      }

    }
    return root.stringType();
  }

  private boolean isInteger(String input) {
    try {
      Integer.parseInt(input);
      return true;
    } catch (NumberFormatException e) {
      return false;
    }
  }

  private boolean isDouble(String input) {
    try {
      Double.parseDouble(input);
      return true;
    } catch (NumberFormatException e) {
      return false;
    }
  }
}
