/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;


import org.mule.metadata.api.model.MetadataType;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Optional;

/**
 * {@link Gson} {@link TypeAdapter<MetadataType>} that helps to serialize and deserialize {@link MetadataType} objects.
 * To use it, register it into {@link Gson} use the {@link GsonBuilder} and invoking
 * {@link GsonBuilder#registerTypeAdapter(Type type, Object typeAdapter)}, with {@link MetadataType} class as the
 * {@code type} parameter and an instance of {@link MetadataTypeGsonTypeAdapter} for the {@code typeAdapter} parameter.
 *
 * @since 1.0
 */
public final class MetadataTypeGsonTypeAdapter extends TypeAdapter<MetadataType>
{

    private final JsonMetadataTypeWriter jsonMetadataTypeWriter = new JsonMetadataTypeWriter();
    private final JsonMetadataTypeLoader jsonMetadataTypeLoader = new JsonMetadataTypeLoader();

    /**
     * Given a {@link MetadataType} as parameter, this serializes it, and writes the Json object using the {@param jsonWriter}
     *
     * @param jsonWriter   {@link JsonWriter} instance to write the serialized object into Json
     * @param metadataType {@link MetadataType} instance to serialize
     * @throws IOException
     */
    @Override
    public void write(JsonWriter jsonWriter, MetadataType metadataType) throws IOException
    {
        jsonMetadataTypeWriter.write(metadataType, jsonWriter);
    }

    /**
     * Uses the {@link JsonReader} containing the JSON representation of a {@link MetadataType}, it gets deserialized and
     * returns the instance.
     *
     * @param jsonReader which contains as root level the JSON representation of a {@link MetadataType}
     * @return a {@link MetadataType} instance that represents the deserialized value of the JSON object.
     * @throws IOException
     */
    @Override
    public MetadataType read(JsonReader jsonReader) throws IOException
    {
        final JsonElement jsonMetadataTypeElement = new JsonParser().parse(jsonReader);
        final Optional<MetadataType> load = jsonMetadataTypeLoader.load(jsonMetadataTypeElement);
        if (load.isPresent())
        {
            return load.get();
        }
        else
        {
            //TODO - review, this case won't never happen
            throw new MetadataSerializingException("Unknown error happen deserializing the MetadataType object");
        }
    }
}