/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.deserializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.ANY;
import static org.mule.metadata.persistence.MetadataTypeConstants.ARRAY;
import static org.mule.metadata.persistence.MetadataTypeConstants.BINARY;
import static org.mule.metadata.persistence.MetadataTypeConstants.BOOLEAN;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE_TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.DICTIONARY;
import static org.mule.metadata.persistence.MetadataTypeConstants.INTERSECTION;
import static org.mule.metadata.persistence.MetadataTypeConstants.NULL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NUMBER;
import static org.mule.metadata.persistence.MetadataTypeConstants.OBJECT;
import static org.mule.metadata.persistence.MetadataTypeConstants.STRING;
import static org.mule.metadata.persistence.MetadataTypeConstants.TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.TUPLE;
import static org.mule.metadata.persistence.MetadataTypeConstants.UNION;

import org.mule.metadata.api.model.MetadataType;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * Provider of {@link TypeDeserializer}s for concrete {@link MetadataType} implementations.
 *
 * @since 1.0
 */
public final class MetadataTypeDeserializerProvider
{

    private final Map<String, TypeDeserializer> typeDeserializers = Collections.unmodifiableMap(new HashMap<String, TypeDeserializer>()
    {{
        put(ANY, new AnyTypeDeserializer());
        put(ARRAY, new ArrayTypeDeserializer());
        put(BOOLEAN, new BooleanTypeDeserializer());
        put(BINARY, new BinaryTypeDeserializer());
        put(DATE, new DateTypeDeserializer());
        put(DATE_TIME, new DateTimeTypeDeserializer());
        put(DICTIONARY, new DictionaryTypeDeserializer());
        put(INTERSECTION, new IntersectionTypeDeserializer());
        put(OBJECT, new ObjectTypeDeserializer());
        put(NUMBER, new NumberTypeDeserializer());
        put(STRING, new StringTypeDeserializer());
        put(TIME, new TimeTypeDeserializer());
        put(TUPLE, new TupleTypeDeserializer());
        put(UNION, new UnionTypeDeserializer());

        put(NULL, (typeObject, baseBuilder, typeLoader) -> baseBuilder.nullType());
    }});

    /**
     * Given a Type name, this provider will return the correspondent deserializer.
     *
     * @param typeName {@link MetadataType} name
     * @return {@link Optional} of the {@link TypeDeserializer}
     */
    public Optional<TypeDeserializer> get(String typeName)
    {
        return Optional.ofNullable(typeDeserializers.get(typeName));
    }
}
