/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.deserializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.ATTRIBUTES;
import static org.mule.metadata.persistence.MetadataTypeConstants.FIELDS;
import static org.mule.metadata.persistence.MetadataTypeConstants.KEY;
import static org.mule.metadata.persistence.MetadataTypeConstants.MODEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NAME;
import static org.mule.metadata.persistence.MetadataTypeConstants.ORDERED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REPEATED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REQUIRED;
import static org.mule.metadata.persistence.deserializer.SerializerUtils.VOLATILE_FORMAT;
import static org.mule.metadata.persistence.deserializer.SerializerUtils.getAnnotations;

import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectKeyBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

/**
 * {@link TypeDeserializer} implementation that deserializes an {@link ObjectType} and returns an {@link ObjectTypeBuilder}
 *
 * @since 1.0
 */
final class ObjectTypeDeserializer extends AbstractTypeDeserializer
{

    ObjectTypeDeserializer()
    {
        super(BaseTypeBuilder::objectType);
    }

    @Override
    public TypeBuilder buildType(JsonObject typeObject, BaseTypeBuilder baseBuilder, JsonMetadataTypeLoader typeLoader)
    {

        final ObjectTypeBuilder objectTypeBuilder = (ObjectTypeBuilder) supplier.get(baseBuilder);
        if (typeObject.has(ORDERED))
        {
            objectTypeBuilder.ordered(typeObject.get(ORDERED).getAsBoolean());
        }
        final JsonElement fields = typeObject.get(FIELDS);
        parseAnnotations(typeObject, objectTypeBuilder);

        for (JsonElement jsonElement : fields.getAsJsonArray())
        {
            final JsonObject field = jsonElement.getAsJsonObject();
            final ObjectFieldTypeBuilder fieldBuilder;
            final JsonObject key = field.get(KEY).getAsJsonObject();

            Optional<ObjectKeyBuilder> keyBuilder = Optional.empty();
            final String keyName = key.get(NAME).getAsString();
            if (isPattern(key))
            {

                fieldBuilder = objectTypeBuilder.addField().key(Pattern.compile(keyName));
            }
            else
            {
                fieldBuilder = objectTypeBuilder.addField();
                keyBuilder = Optional.of(fieldBuilder.key(parseKey(keyName)));
            }

            getAnnotations(key).forEach(fieldBuilder::withKeyAnnotation);
            getAnnotations(field).forEach(fieldBuilder::with);
            final JsonElement valueElement = field.get(MODEL);
            if (valueElement.isJsonObject())
            {
                final JsonObject valueObject = valueElement.getAsJsonObject();
                if (valueObject.has(ATTRIBUTES) && keyBuilder.isPresent())
                {
                    parseAttributes(valueObject, keyBuilder.get(), typeLoader, VOLATILE_FORMAT);
                }
                if (valueObject.has(REQUIRED))
                {
                    fieldBuilder.required(valueObject.get(REQUIRED).getAsBoolean());
                }
                if (valueObject.has(REPEATED))
                {
                    fieldBuilder.repeated(valueObject.get(REPEATED).getAsBoolean());
                }

                fieldBuilder.value(typeLoader.buildType(valueObject, new BaseTypeBuilder(VOLATILE_FORMAT)));

            }
            else
            {
                fieldBuilder.value(typeLoader.buildType(valueElement, fieldBuilder.value()));
            }
        }
        return objectTypeBuilder;
    }

    private void parseAttributes(JsonObject value, ObjectKeyBuilder key, JsonMetadataTypeLoader typeLoader, MetadataFormat format)
    {
        final JsonObject attributes = value.get(ATTRIBUTES).getAsJsonObject();

        attributes.entrySet().forEach(entry -> key
                .addAttribute()
                .name(parseKey(entry.getKey()))
                .value(typeLoader.buildType(entry.getValue(), new BaseTypeBuilder(format))));
    }

    private QName parseKey(String key)
    {
        final Pattern compile = Pattern.compile("(?:\\{([\\d\\w\\D\\W]*)\\})?[(\\d\\w\\D\\W)]*");
        final Matcher matcher = compile.matcher(key);
        if (matcher.find())
        {
            if (matcher.groupCount() == 1)
            {
                return new QName(key);
            }
            if (matcher.groupCount() == 2)
            {
                return new QName(matcher.group(0), matcher.group(1));
            }
        }
        return new QName(key);
    }

    private boolean isPattern(JsonElement value)
    {
        if (value.isJsonObject())
        {
            for (TypeAnnotation typeAnnotation : getAnnotations(value.getAsJsonObject()))
            {
                if (typeAnnotation.getName().equals(RegexPatternAnnotation.NAME))
                {
                    return true;
                }
            }
        }

        return false;
    }
}
