/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.type.adapter;

import org.mule.metadata.api.annotation.TypeAnnotation;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.reflect.TypeToken;

/**
 * {@link TypeAdapterFactory} implementation which creates instances of {@link OnlyOneFieldTypeAdapter} when a new object
 * is required to be serialized and that type is a registered {@link TypeAnnotation}.
 *
 * @since 1.0
 */
public final class OnlyOneFieldTypeAdapterFactory implements TypeAdapterFactory
{

    private final Class<? extends TypeAnnotation>[] typeAnnotations;

    public OnlyOneFieldTypeAdapterFactory(Class<? extends TypeAnnotation>... typeAnnotations)
    {
        this.typeAnnotations = typeAnnotations;
    }

    @Override
    public <T> TypeAdapter<T> create(Gson gson, TypeToken<T> type)
    {
        if (isRegisteredTypeAnnotation(type))
        {
            return new OnlyOneFieldTypeAdapter<>(gson, (Class<T>) type.getRawType());
        }
        return null;
    }

    private <T> boolean isRegisteredTypeAnnotation(TypeToken<T> type)
    {
        for (Class<? extends TypeAnnotation> typeAnnotation : typeAnnotations)
        {
            if (type.getRawType().isAssignableFrom(typeAnnotation))
            {
                return true;
            }
        }
        return false;
    }
}
