/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.ObjectType;

import com.google.gson.stream.JsonWriter;

import java.util.Optional;

/**
 * Allows to serialize and deserealize {@link ObjectType} instances
 * which are not explicitly defined but instead pointed at by some means
 *
 * @since 1.0
 */
public interface ObjectTypeReferenceHandler {

  /**
   * Uses the {@code writer} to write a reference to the given {@code type}.
   * <p>
   * This method returns the written reference, so that it can be used as an id
   * to deserialize. However, some implementations might be unable to handle
   * some specific types, which is why the reference might be {@link Optional#empty()}.
   * <p>
   * In such case, it's the invoker responsibility to either attempt using a different
   * implementation of this contract or to just treat the type explicitly instead of
   * through a reference.
   *
   * @param type   the type to write
   * @param writer a {@link JsonWriter}
   * @return the reference that was written or {@link Optional#empty()}
   */
  Optional<String> writeReference(ObjectType type, JsonWriter writer);

  /**
   * Returns a {@link TypeBuilder} referenced by the given {@code typeId}.
   * If the reference cannot be handled or points to an unknown type it returns
   * {@link Optional#empty()}
   *
   * @param typeId a type id
   * @return an {@link Optional} {@link TypeBuilder}
   */
  Optional<TypeBuilder> readReference(String typeId);

}
