/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.annotation.AnnotationsRegistry;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.api.annotation.FieldOccurrenceAnnotation;
import org.mule.metadata.api.annotation.IntAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.annotation.NumberRangeAnnotation;
import org.mule.metadata.api.annotation.OpenObjectAnnotation;
import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.annotation.UniquesItemsAnnotation;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;

import java.util.Map;
import java.util.ServiceLoader;

/**
 * Helper class that maps the {@link TypeAnnotation#getName()} of each {@link TypeAnnotation} with the concrete implementation
 * class. This mapping is useful when serializing a {@link TypeAnnotation} with out loosing the reference to this class
 * to be able at deserializing time to know which class instantiate and initialize.
 * <p>
 * By default this mapping provides all the out of the box {@link TypeAnnotation} implementation like {@link LabelAnnotation},
 * {@link DefaultValueAnnotation}, etc, but this mapping is extensible to provide more mappings using SPI (Service Provider
 * Interface) loading defined {@link AnnotationsRegistry} implementations.
 *
 * @see AnnotationsRegistry
 * @since 1.0
 */
final class TypeAnnotationNameClassMapping {

  private final HashBiMap<String, Class<? extends TypeAnnotation>> stringClassHashMap = HashBiMap.create();

  TypeAnnotationNameClassMapping() {
    stringClassHashMap.put(LabelAnnotation.NAME, LabelAnnotation.class);
    stringClassHashMap.put(LengthAnnotation.NAME, LengthAnnotation.class);
    stringClassHashMap.put(AccessibilityAnnotation.NAME, AccessibilityAnnotation.class);
    stringClassHashMap.put(DefaultValueAnnotation.NAME, DefaultValueAnnotation.class);
    stringClassHashMap.put(DescriptionAnnotation.NAME, DescriptionAnnotation.class);
    stringClassHashMap.put(EnumAnnotation.NAME, EnumAnnotation.class);
    stringClassHashMap.put(ExampleAnnotation.NAME, ExampleAnnotation.class);
    stringClassHashMap.put(FieldOccurrenceAnnotation.NAME, FieldOccurrenceAnnotation.class);
    stringClassHashMap.put(IntAnnotation.NAME, IntAnnotation.class);
    stringClassHashMap.put(NumberRangeAnnotation.NAME, NumberRangeAnnotation.class);
    stringClassHashMap.put(OpenObjectAnnotation.NAME, OpenObjectAnnotation.class);
    stringClassHashMap.put(RegexPatternAnnotation.NAME, RegexPatternAnnotation.class);
    stringClassHashMap.put(TypeIdAnnotation.NAME, TypeIdAnnotation.class);
    stringClassHashMap.put(UniquesItemsAnnotation.NAME, UniquesItemsAnnotation.class);
    stringClassHashMap.put(ClassInformationAnnotation.NAME, ClassInformationAnnotation.class);

    ServiceLoader<AnnotationsRegistry> annotationsRegistries = ServiceLoader.load(AnnotationsRegistry.class);
    annotationsRegistries.forEach((consumer) -> consumer.getRegistry().forEach(stringClassHashMap::put));
  }

  private BiMap<String, Class<? extends TypeAnnotation>> getMapping() {
    return stringClassHashMap;
  }

  /**
   * @return A {@link Map} that links a friendly name with a {@link TypeAnnotation} class.
   * This is helpful when deserializing a {@link TypeAnnotation} to be able to create the correct instance type.
   */
  Map<String, Class<? extends TypeAnnotation>> getNameClassMapping() {
    return getMapping();
  }

  /**
   * @return A {@link Map} that links a {@link TypeAnnotation} class with their friendly name.
   * This is helpful when serializing a {@link TypeAnnotation} to be able to tag it with a more friendly name that
   * the full qualifier name of the class.
   */
  Map<Class<? extends TypeAnnotation>, String> getClassNameMapping() {
    return getMapping().inverse();
  }

}
