/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.type.adapter;

import com.google.gson.Gson;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.util.Optional;

/**
 * {@link TypeAdapter<Optional>} which overrides the default behaviour of {@link Gson} to be able to write more clean
 * and less verbose JSON.
 * This {@link TypeAdapter} unwraps the Optional value and serializes it, when deserializing the value gets wrapped
 * to comply with the Java types.
 * <p>
 * <b>Using default GSON TypeAdapter:</b>
 * <pre>{"optionalValue":{"value":"this is the value"}}</pre>
 *
 * <b>Using OptionalTypeAdapter:</b>
 * <pre>{"optionalValue":"this is the value"}</pre>
 *
 * @param <T> Type of the Optional value
 * @since 1.0
 */
final class OptionalTypeAdapter<T> extends TypeAdapter<Optional<T>> {

  private final TypeToken type;
  private final Gson gson;

  OptionalTypeAdapter(TypeToken<T> type, Gson gson) {
    this.type = type;
    this.gson = gson;
  }

  @Override
  public void write(JsonWriter out, Optional<T> value) throws IOException {
    if (value.isPresent()) {
      gson.getAdapter(new TypeToken<T>() {}).write(out, value.get());
    } else {
      out.beginObject().endObject();
    }
  }

  @Override
  public Optional<T> read(JsonReader in) throws IOException {
    final T t = gson.getAdapter(new TypeToken<T>() {}).fromJson(new JsonParser().parse(in).toString());

    return Optional.ofNullable(t);
  }
}
