/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.AttributeFieldTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectKeyBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.xml.handler.HandlerManager;
import org.mule.metadata.xml.utils.SchemaHelper;

import java.io.File;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.xml.namespace.QName;

import org.apache.xerces.impl.dv.xs.XSSimpleTypeDecl;
import org.apache.xerces.impl.xs.XSComplexTypeDecl;
import org.apache.xerces.xs.XSAnnotation;
import org.apache.xerces.xs.XSAttributeDeclaration;
import org.apache.xerces.xs.XSAttributeUse;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSTerm;
import org.apache.xerces.xs.XSTypeDefinition;
import org.apache.xerces.xs.XSWildcard;
import org.w3c.dom.TypeInfo;

public class XmlTypeLoader implements TypeLoader
{

    public static final String TEXT_CONTENT_FIELD_NAME = "text()";
    public static MetadataFormat XML = new MetadataFormat("XML", "xml", "text/xml", "application/xml");

    private HandlerManager handlerManager;
    private ModelFactory modelFactory;
    private Map<QName, TypeBuilder<?>> types;

    public XmlTypeLoader(List<File> schemas)
    {
        this.modelFactory = ModelFactory.fromSchemas(schemas);
        this.types = new HashMap<>();
        this.handlerManager = new HandlerManager();
    }

    public XmlTypeLoader(ModelFactory modelFactory)
    {
        this.modelFactory = modelFactory;
        this.types = new HashMap<>();
        this.handlerManager = new HandlerManager();
    }

    @Override
    public Optional<MetadataType> load(String typeIdentifier)
    {
        try
        {
            final QName rootElement = QName.valueOf(typeIdentifier);
            final XSModel model = modelFactory.getModel();
            if (model != null)
            {
                final XSElementDeclaration elementDeclaration = model.getElementDeclaration(rootElement.getLocalPart(), rootElement.getNamespaceURI());
                if (elementDeclaration == null)
                {
                    return Optional.empty();
                }
                final BaseTypeBuilder<?> rootBuilder = BaseTypeBuilder.create(XML);
                // Create root object with the root name
                final ObjectTypeBuilder<?> rootObject = rootBuilder.objectType();
                rootObject.id(typeIdentifier);
                modelFactory.getExample().ifPresent(rootObject::with);
                final ObjectFieldTypeBuilder<?> field = rootObject.addField();
                final ObjectKeyBuilder<?> key = field.key(rootElement);

                processAttributes(elementDeclaration, key);
                final BaseTypeBuilder<?> valueBuilder = field.value();
                processValue(elementDeclaration, valueBuilder);
                return Optional.ofNullable(rootBuilder.build());
            }
            else
            {
                return Optional.empty();
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    private void processValue(XSElementDeclaration elementDeclaration, BaseTypeBuilder<?> typeBuilder)
    {
        final XSTypeDefinition typeDefinition = elementDeclaration.getTypeDefinition();
        if (typeDefinition instanceof XSComplexTypeDecl)
        {
            final XSComplexTypeDecl complexType = (XSComplexTypeDecl) typeDefinition;
            final Optional<Short> groupType = getGroupType(complexType);
            if (groupType.isPresent() && groupType.get().equals(XSModelGroup.COMPOSITOR_CHOICE))
            {
                processUnionType(typeBuilder, complexType);
            }
            else
            {
                if (XSComplexTypeDefinition.CONTENTTYPE_SIMPLE == complexType.getContentType())
                {
                    typeBuilder.stringType();
                }
                else
                {
                    final ObjectTypeBuilder<? extends BaseTypeBuilder<?>> objectType = typeBuilder.objectType();
                    if (XSComplexTypeDefinition.CONTENTTYPE_MIXED == complexType.getContentType())
                    {
                        objectType.addField().key(TEXT_CONTENT_FIELD_NAME).value().stringType();
                    }
                    final Optional<QName> typeName = getTypeName(typeDefinition);
                    if (typeName.isPresent())
                    {
                        objectType.id(typeName.get().toString());
                        types.put(typeName.get(), objectType);
                    }
                    processAnnotations(complexType, objectType);
                    processElements(complexType, objectType);
                }
            }
        }
        else if (typeDefinition instanceof XSSimpleTypeDecl)
        {
            handlerManager.handle((XSSimpleTypeDecl) typeDefinition, SchemaHelper.getDefaultValue(elementDeclaration), typeBuilder);
        }
    }

    private void processUnionType(BaseTypeBuilder<?> typeBuilder, XSComplexTypeDecl complexType)
    {
        final UnionTypeBuilder<?> unionType = typeBuilder.unionType();
        final XSParticle particle = complexType.getParticle();
        if (particle != null)
        {
            final XSTerm term = particle.getTerm();
            if (term instanceof XSModelGroup)
            {
                final XSModelGroup xsModelGroup = (XSModelGroup) term;
                final XSObjectList particles = xsModelGroup.getParticles();
                for (Object xsObject : particles)
                {
                    if (xsObject instanceof XSParticle)
                    {
                        final XSParticle part = (XSParticle) xsObject;
                        final XSTerm element = part.getTerm();
                        if (element instanceof XSElementDeclaration)
                        {
                            final ObjectTypeBuilder<?> objectType = unionType.of().objectType();
                            final XSElementDeclaration elementDeclaration = (XSElementDeclaration) element;
                            final String name = element.getName();
                            final String namespace = element.getNamespace();
                            final ObjectFieldTypeBuilder<?> field = objectType.addField();
                            final ObjectKeyBuilder<? extends ObjectFieldTypeBuilder<?>> key = field.key(new QName(namespace, name));
                            processAttributes(elementDeclaration, key);
                            field.required(part.getMinOccurs() > 0);
                            final BaseTypeBuilder<?> fieldValue = field.value();
                            processValue(elementDeclaration, fieldValue);
                        }
                    }
                }
            }
        }
    }

    private Optional<Short> getGroupType(XSComplexTypeDecl complexType)
    {
        final XSParticle particle = complexType.getParticle();
        if (particle != null)
        {
            final XSTerm term = particle.getTerm();
            if (term instanceof XSModelGroup)
            {
                return Optional.of(((XSModelGroup) term).getCompositor());
            }
        }
        return Optional.empty();
    }

    private void processAnnotations(XSComplexTypeDecl complexType, ObjectTypeBuilder<?> objectType)
    {
        final XSObjectList annotations = complexType.getAnnotations();
        for (Object annotation : annotations)
        {
            final XSAnnotation xsAnnotation = (XSAnnotation) annotation;
            final Collection<SchemaHelper.XmlDoc> documentation = SchemaHelper.getDocumentation(xsAnnotation.getAnnotationString());
            for (SchemaHelper.XmlDoc xmlDoc : documentation)
            {
                objectType.description(xmlDoc.getLang().orElse(null), xmlDoc.getContent());
            }
        }
    }

    private void processAttributes(XSElementDeclaration elementDeclaration, ObjectKeyBuilder<?> keyBuilder)
    {
        final XSTypeDefinition typeDefinition = elementDeclaration.getTypeDefinition();
        if (typeDefinition instanceof XSComplexTypeDecl)
        {
            final XSObjectList attributeUses = ((XSComplexTypeDefinition) typeDefinition).getAttributeUses();
            for (Object attributeUse : attributeUses)
            {
                if (attributeUse instanceof XSAttributeUse)
                {
                    final XSAttributeUse xsAttributeUse = (XSAttributeUse) attributeUse;
                    final XSAttributeDeclaration attrDeclaration = xsAttributeUse.getAttrDeclaration();
                    final String localName = attrDeclaration.getName();
                    final String namespace = attrDeclaration.getNamespace();
                    final AttributeFieldTypeBuilder<?> attributeField = keyBuilder.addAttribute();
                    attributeField.name(new QName(namespace, localName));
                    attributeField.required(xsAttributeUse.getRequired());
                    handlerManager.handle((XSSimpleTypeDecl) attrDeclaration.getTypeDefinition(), SchemaHelper.getDefaultValue(xsAttributeUse), attributeField.value());
                }
            }
        }
    }

    private void processElements(XSComplexTypeDecl complexType, ObjectTypeBuilder<?> objectType)
    {
        final XSParticle particle = complexType.getParticle();
        if (particle != null)
        {
            final XSTerm term = particle.getTerm();
            if (term instanceof XSModelGroup)
            {
                final XSModelGroup xsModelGroup = (XSModelGroup) term;
                objectType.ordered(xsModelGroup.getCompositor() == XSModelGroup.COMPOSITOR_SEQUENCE);
                final XSObjectList particles = xsModelGroup.getParticles();
                for (Object xsObject : particles)
                {
                    if (xsObject instanceof XSParticle)
                    {
                        final XSParticle part = (XSParticle) xsObject;
                        final XSTerm element = part.getTerm();
                        if (element instanceof XSElementDeclaration)
                        {
                            final XSElementDeclaration xsElementDeclaration = (XSElementDeclaration) element;
                            final String name = element.getName();
                            final String namespace = element.getNamespace();
                            final ObjectFieldTypeBuilder<? extends ObjectTypeBuilder<?>> field = objectType.addField();
                            field.key(new QName(namespace, name));
                            field.required(part.getMinOccurs() > 0);
                            field.repeated(isRepeated(part));

                            Optional<Number> min = Optional.empty();
                            Optional<Number> max = Optional.empty();

                            if (part.getMinOccurs() > 1)
                            {
                                min = Optional.of(part.getMinOccurs());
                            }

                            if (part.getMaxOccurs() > 1)
                            {
                                max = Optional.of(part.getMaxOccurs());
                            }

                            if (min.isPresent() || max.isPresent())
                            {
                                field.occurrence(min, max);
                            }

                            final XSTypeDefinition typeDefinition = ((XSElementDeclaration) element).getTypeDefinition();
                            final Optional<QName> typeName = getTypeName(typeDefinition);
                            if (typeName.isPresent() && types.containsKey(typeName.get()))
                            {
                                field.value(types.get(typeName.get()));
                            }
                            else
                            {
                                final BaseTypeBuilder<?> fieldValue = field.value();
                                processValue(xsElementDeclaration, fieldValue);
                            }
                        }
                        else if (element instanceof XSWildcard)
                        {
                            if (((XSWildcard) element).getConstraintType() == XSWildcard.NSCONSTRAINT_ANY)
                            {
                                objectType.open();
                            }
                        }
                    }
                }
            }
        }
    }

    private Optional<QName> getTypeName(XSTypeDefinition typeDefinition)
    {
        Optional<QName> typeQname = Optional.empty();
        if (typeDefinition instanceof TypeInfo)
        {
            final TypeInfo definition = (TypeInfo) typeDefinition;
            if (definition.getTypeName() != null)
            {
                typeQname = Optional.of(new QName(definition.getTypeNamespace(), definition.getTypeName()));
            }
        }
        return typeQname;
    }

    private boolean isRepeated(XSParticle part)
    {
        return part.getMaxOccursUnbounded() || part.getMaxOccurs() > 1 || part.getMinOccurs() > 1;
    }

}
