/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.handler;

import org.apache.xerces.impl.dv.xs.XSSimpleTypeDecl;

import org.apache.xerces.xs.StringList;
import org.apache.xerces.xs.XSMultiValueFacet;
import org.apache.xerces.xs.XSObject;
import org.apache.xerces.xs.XSSimpleTypeDefinition;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.StringTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.xml.utils.SchemaHelper;
import org.mule.metadata.xml.utils.XmlConstants;

import javax.xml.namespace.QName;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

public class StringHandler implements XmlHandler
{

    public static final List<QName> STRING_TYPES = Arrays.asList(XmlConstants.XSD_STRING, XmlConstants.XSD_ANY, XmlConstants.XSD_NORMALIZEDSTRING);

    @Override
    public boolean handles(QName schema)
    {
        return STRING_TYPES.contains(schema);
    }

    @Override
    public TypeBuilder<?> handle(XSSimpleTypeDecl element, Optional<String> defaultValue, BaseTypeBuilder<?> builder)
    {
        final StringTypeBuilder<?> stringType = builder.stringType();
        final StringList enumeration = element.getLexicalEnumeration();
        if (!enumeration.isEmpty())
        {
            final String[] values = (String[]) enumeration.toArray(new String[enumeration.size()]);
            stringType.enumOf(values);
        }
        else
        {
            if (defaultValue.isPresent() && !defaultValue.get().isEmpty())
            {
                stringType.defaultValue(defaultValue.get());
            }

            // Pattern
            final XSObject patternFacetObject = element.getFacet(XSSimpleTypeDefinition.FACET_PATTERN);
            if (patternFacetObject instanceof XSMultiValueFacet)
            {
                XSMultiValueFacet patternFacet = (XSMultiValueFacet) patternFacetObject;
                final StringList lexicalFacetValues = patternFacet.getLexicalFacetValues();
                if (lexicalFacetValues != null && !lexicalFacetValues.isEmpty())
                {
                    stringType.pattern(lexicalFacetValues.item(0));
                }
            }

            // Boundary
            final Optional<Integer> minLength = Optional.ofNullable(element.getFacet(XSSimpleTypeDefinition.FACET_MINLENGTH))
                                                        .map(SchemaHelper::getIntegerValue);
            final Optional<Integer> maxLength = Optional.ofNullable(element.getFacet(XSSimpleTypeDefinition.FACET_MAXLENGTH))
                                                        .map(SchemaHelper::getIntegerValue);
            if (minLength.isPresent() || maxLength.isPresent())
            {
                stringType.boundary(minLength, maxLength);
            }

            // Length
            final Optional<Integer> length = Optional.ofNullable(element.getFacet(XSSimpleTypeDefinition.FACET_LENGTH))
                                                     .map(SchemaHelper::getIntegerValue);
            if (length.isPresent())
            {
                stringType.length(length.get());
            }

        }
        return stringType;
    }
}
