/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml;

import static java.lang.String.format;
import static java.util.stream.Collectors.toMap;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.xml.utils.SchemaHelper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URL;
import java.nio.file.Files;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.xerces.dom.DOMInputImpl;
import org.apache.xerces.impl.xs.XSImplementationImpl;
import org.apache.xerces.impl.xs.util.LSInputListImpl;
import org.apache.xerces.xs.XSLoader;
import org.apache.xerces.xs.XSModel;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;

public class ModelFactory {

  private XSModel model;
  private Optional<ExampleAnnotation> example;

  private ModelFactory(XSModel model, ExampleAnnotation example) {
    this.model = model;
    this.example = Optional.ofNullable(example);
  }

  public XSModel getModel() {
    return model;
  }

  public Optional<ExampleAnnotation> getExample() {
    return example;
  }

  public static ModelFactory fromExample(File exampleFile) {
    try {
      return fromExample(Files.readAllLines(exampleFile.toPath()).stream().collect(Collectors.joining()));
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromExample(String exampleXML) {

    try {
      final XSLoader schemaLoader = initializeXSLoader();
      List<String> schemas = SchemaHelper.generateXSD(exampleXML);

      final DOMInputImpl[] domInputs = schemas.stream()
          .map((schema) -> new DOMInputImpl(null, null, null, new StringReader(schema), "UTF-8")).toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      return new ModelFactory(model, new ExampleAnnotation(StringEscapeUtils.escapeXml11(exampleXML)));
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public static ModelFactory fromSchemas(List<File> schemas) {

    Function<File, InputStream> fileToInputStream = s -> {
      try {
        return new FileInputStream(s);
      } catch (FileNotFoundException e) {
        throw new RuntimeException(format("Error while getting file: [%s] input stream", s.getName()), e);
      }
    };

    return createModelFactory(schemas.stream().collect(toMap(s -> s.toURI().toString(), fileToInputStream)));

  }

  public static ModelFactory fromSchemas(Set<String> schemas) {
    Function<String, InputStream> urlToInputStream = s -> {
      try {
        return new URL(s).openStream();
      } catch (IOException e) {
        throw new RuntimeException(format("Error while getting schema url: [%s] input stream", s), e);
      }
    };

    return createModelFactory(schemas.stream().collect(toMap(s -> s, urlToInputStream)));
  }

  private static ModelFactory createModelFactory(Map<String, InputStream> schemasMap) {
    try {
      final XSLoader schemaLoader = initializeXSLoader();
      final DOMInputImpl[] domInputs = schemasMap.entrySet()
          .stream()
          .map((schema) -> new DOMInputImpl(null, schema.getKey(), null, schema.getValue(), "UTF-8"))
          .toArray(DOMInputImpl[]::new);
      final XSModel model = schemaLoader.loadInputList(new LSInputListImpl(domInputs, domInputs.length));
      return new ModelFactory(model, null);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static XSLoader initializeXSLoader() throws ClassNotFoundException, InstantiationException, IllegalAccessException {
    final DOMImplementationRegistry registry;
    registry = DOMImplementationRegistry.newInstance();
    final XSImplementationImpl impl = (XSImplementationImpl) registry.getDOMImplementation("XS-Loader");
    return impl.createXSLoader(null);
  }
}
