/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.xml.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.NumberTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.xml.utils.SchemaHelper;
import org.mule.metadata.xml.utils.XmlConstants;

import java.util.Arrays;
import java.util.Optional;

import javax.xml.namespace.QName;

import org.apache.xerces.impl.dv.xs.XSSimpleTypeDecl;
import org.apache.xerces.xs.XSSimpleTypeDefinition;

public class NumberHandler implements XmlHandler {

  public static final java.util.List<QName> NUMBER_TYPES =
      Arrays.asList(XmlConstants.XSD_BYTE, XmlConstants.XSD_INT, XmlConstants.XSD_INTEGER, XmlConstants.XSD_SHORT,
                    XmlConstants.XSD_NEGATIVEINTEGER, XmlConstants.XSD_NONNEGATIVEINTEGER,
                    XmlConstants.XSD_NONPOSITIVEINTEGER, XmlConstants.XSD_POSITIVEINTEGER, XmlConstants.XSD_UNSIGNEDBYTE,
                    XmlConstants.XSD_UNSIGNEDINT, XmlConstants.XSD_UNSIGNEDSHORT, XmlConstants.XSD_LONG,
                    XmlConstants.XSD_UNSIGNEDLONG,
                    XmlConstants.XSD_FLOAT, XmlConstants.XSD_DOUBLE, XmlConstants.XSD_DECIMAL);

  @Override
  public boolean handles(QName schema) {
    return NUMBER_TYPES.contains(schema);
  }

  @Override
  public TypeBuilder<?> handle(XSSimpleTypeDecl element, Optional<String> defaultValue, BaseTypeBuilder<?> builder) {
    final NumberTypeBuilder<? extends BaseTypeBuilder<?>> numberType = builder.numberType();

    if (defaultValue.isPresent() && !defaultValue.get().isEmpty()) {
      numberType.defaultValue(defaultValue.get());
    }

    // Boundary
    final Optional<Number> minLength = Optional.ofNullable(element.getFacet(XSSimpleTypeDefinition.FACET_MININCLUSIVE))
        .map(SchemaHelper::getDoubleValue);
    final Optional<Number> maxLength = Optional.ofNullable(element.getFacet(XSSimpleTypeDefinition.FACET_MAXINCLUSIVE))
        .map(SchemaHelper::getDoubleValue);

    if (minLength.isPresent() || maxLength.isPresent()) {
      numberType.range(minLength, maxLength);
    }
    return numberType;
  }
}
